<?php
/**
 * Banner Admin Model
 *
 * @category   GadgetModel
 * @package    Banner
 * @author     Jonathan Hernandez <ion@suavizado.com>
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @author     Jon Wood <jon@jellybob.co.uk>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2004-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/gpl.html
 */
require_once JAWS_PATH . 'gadgets/Banner/Model.php';

class BannerAdminModel extends BannerModel
{
    var $_Name = 'Banner';

    /**
     * Installs the gadget
     *
     * @access       public
     * @return       true on successful installation, Jaws_Error otherwise
     */
    function InstallGadget()
    {
        if (!Jaws_Utils::is_writable(JAWS_DATA)) {
            return new Jaws_Error(_t('GLOBAL_ERROR_FAILED_DIRECTORY_UNWRITABLE', JAWS_DATA));
        }

        $new_dir = JAWS_DATA . 'banners' . DIRECTORY_SEPARATOR;
        if (!Jaws_Utils::mkdir($new_dir)) {
            return new Jaws_Error(_t('GLOBAL_ERROR_FAILED_CREATING_DIR', $new_dir), _t('BANNER_NAME'));
        }

        $result = $this->installSchema('schema.xml');
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        $result = $this->installSchema('insert.xml', '', 'schema.xml', true);
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        //registry keys.

        return true;
    }

    /**
     * Uninstall the gadget
     *
     * @access  public
     * @return  boolean True on a successful one and Jaws_Error otherwise
     */
    function UninstallGadget()
    {
        $tables = array('banners',
                        'banners_groups');
        foreach ($tables as $table) {
            $result = $GLOBALS['db']->dropTable($table);
            if (Jaws_Error::IsError($result)) {
                $gName  = _t('BANNER_NAME');
                $errMsg = _t('GLOBAL_ERROR_GADGET_NOT_UNINSTALLED', $gName);
                $GLOBALS['app']->Session->PushLastResponse($errMsg, RESPONSE_ERROR);
                return new Jaws_Error($errMsg, $gName);
            }
        }

        //registry keys

        return true;
    }

    /**
     * Update the gadget
     *
     * @access  public
     * @param   string  $old    Current version (in registry)
     * @param   string  $new    New version (in the $gadgetInfo file)
     * @return  boolean  Success/Failure (Jaws_Error)
     */
    function UpdateGadget($old, $new)
    {
        $result = $this->installSchema('schema.xml', '', "$old.xml");
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        $result = $this->installSchema('insert.xml', '', 'schema.xml', true);
        if (Jaws_Error::IsError($result)) {
            // maybe user hasve banner group with this name
            //return $result;
        }

        // Registry keys.

        return true;
    }

    /**
    * Insert a bannser
    * @access  public
    *
    * @return  boolean Success/Failure (Jaws_Error)
    */
    function InsertBanner($title, $url, $gid, $direction, $banner, $template, $views_limit,
                          $clicks_limit, $start_time, $stop_time, $random, $visible)
    {
        $sql = 'SELECT COUNT([id]) FROM [[banners]] WHERE [title] = {title}';
        $bc = $GLOBALS['db']->queryOne($sql, array('title' => $title));
        if (Jaws_Error::IsError($bc)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        if ($bc > 0) {
            $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_BANNERS_ERROR_TITLE_DUPLICATE'), RESPONSE_ERROR);
            return false;
        }

        $sql = '
            INSERT INTO [[banners]]
                ([title], [url], [gid], [banner], [template], [direction], [views], [views_limitation],
                [clicks], [clicks_limitation], [start_time], [stop_time], [createtime], [updatetime],
                [random], [visible])
            VALUES
                ({title}, {url}, {gid}, {banner}, {template}, {direction}, 0, {views_limit}, 0,
                {clicks_limit}, {start_time}, {stop_time},{now}, {now}, {random}, {visible})';

        MDB2::loadFile('Date');
        $date = $GLOBALS['app']->loadDate();
        $xss  = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $params                 = array();
        $params['title']        = $xss->parse($title);
        $params['url']          = $url;
        $params['gid']          = ((empty($gid) || !is_numeric($gid)) ? 0: $gid);
        $params['banner']       = $banner;
        $params['template']     = $xss->parse($template, false);
        $params['direction']    = $direction;
        $params['views_limit']  = ((empty($views_limit)  || !is_numeric($views_limit)) ? 0: $views_limit);
        $params['clicks_limit'] = ((empty($clicks_limit) || !is_numeric($clicks_limit))? 0: $clicks_limit);
        $params['start_time']   = ($date->ValidDBDate($start_time)? $start_time : null);
        $params['stop_time']    = ($date->ValidDBDate($stop_time)? $stop_time : null);
        $params['now']          = MDB2_Date::mdbNow();
        $params['random']       = $random;
        $params['visible']      = $visible;
        $res = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }
        $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_BANNERS_CREATED', $title), RESPONSE_NOTICE);

        return true;
    }

    /**
    * Update a bannser
    * @access  public
    *
    * @return  boolean Success/Failure (Jaws_Error)
    */
    function UpdateBanner($bid, $title, $url, $gid, $direction, $banner, $template, $views_limit,
                           $clicks_limit, $start_time, $stop_time, $random, $visible)
    {
        $sql = 'SELECT COUNT([id]) FROM [[banners]] WHERE [id] != {bid} AND [title] = {title}';
        $bc = $GLOBALS['db']->queryOne($sql, array('bid' => $bid, 'title' => $title));
        if (Jaws_Error::IsError($bc)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        if ($bc > 0) {
            $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_BANNERS_ERROR_TITLE_DUPLICATE'), RESPONSE_ERROR);
            return false;
        }

        $sql = '
            UPDATE [[banners]] SET
                [title]             = {title},
                [url]               = {url},
                [gid]               = {gid},
                [banner]            = {banner},
                [template]          = {template},
                [direction]         = {direction},
                [views_limitation]  = {views_limit},
                [clicks_limitation] = {clicks_limit},
                [start_time]        = {start_time},
                [stop_time]         = {stop_time},
                [updatetime]        = {now},
                [random]            = {random},
                [visible]           = {visible}
            WHERE [id] = {bid}';

        MDB2::loadFile('Date');
        $date = $GLOBALS['app']->loadDate();
        $xss  = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $params                 = array();
        $params['bid']          = $bid;
        $params['title']        = $xss->parse($title);
        $params['url']          = $url;
        $params['gid']          = ((empty($gid) || !is_numeric($gid)) ? 0: $gid);
        $params['banner']       = $banner;
        $params['template']     = $xss->parse($template, false);
        $params['direction']    = $direction;
        $params['views_limit']  = ((empty($views_limit)  || !is_numeric($views_limit)) ? 0: $views_limit);
        $params['clicks_limit'] = ((empty($clicks_limit) || !is_numeric($clicks_limit))? 0: $clicks_limit);
        $params['start_time']   = ($date->ValidDBDate($start_time)? $start_time : null);
        $params['stop_time']    = ($date->ValidDBDate($stop_time)? $stop_time : null);
        $params['now']          = MDB2_Date::mdbNow();
        $params['random']       = $random;
        $params['visible']      = $visible;
        $res = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_BANNERS_UPDATED', $title), RESPONSE_NOTICE);
        return true;
    }

    /**
    * Insert a group
    * @access  public
    *
    * @return  boolean Success/Failure (Jaws_Error)
    */
    function InsertGroup($title, $limit_count, $visible)
    {
        $sql = 'SELECT COUNT([id]) FROM [[banners_groups]] WHERE [title] = {title}';
        $gc = $GLOBALS['db']->queryOne($sql, array('title' => $title));
        if (Jaws_Error::IsError($gc)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        if ($gc > 0) {
            $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_GROUPS_ERROR_TITLE_DUPLICATE'), RESPONSE_ERROR);
            return false;
        }

        $sql = '
            INSERT INTO [[banners_groups]]
                ([title], [limit_count], [visible])
            VALUES
                ({title}, {limit_count}, {visible})';

        $xss  = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $params                = array();
        $params['title']       = $xss->parse($title);
        $params['limit_count'] = ((empty($limit_count)  || !is_numeric($limit_count)) ? 0: $limit_count);
        $params['visible']     = $visible;
        $res = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }
        $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_GROUPS_CREATED', $title), RESPONSE_NOTICE);

        return true;
    }

    /**
    * Update a group
    * @access  public
    *
    * @return  boolean Success/Failure (Jaws_Error)
    */
    function UpdateGroup($gid, $title, $limit_count, $visible)
    {
        $sql = 'SELECT COUNT([id]) FROM [[banners_groups]] WHERE [id] != {gid} AND [title] = {title}';
        $gc = $GLOBALS['db']->queryOne($sql, array('gid' => $gid, 'title' => $title));
        if (Jaws_Error::IsError($gc)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        if ($gc > 0) {
            $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_GROUPS_ERROR_TITLE_DUPLICATE'), RESPONSE_ERROR);
            return false;
        }

        $sql = '
            UPDATE [[banners_groups]] SET
                [title]       = {title},
                [limit_count] = {limit_count},
                [visible]     = {visible}
            WHERE [id] = {id}';

        $xss  = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $params                 = array();
        $params['id']           = $gid;
        $params['title']        = $xss->parse($title);
        $params['limit_count']  = ((empty($limit_count)  || !is_numeric($limit_count)) ? 0: $limit_count);
        $params['visible'] = $visible;
        $res = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }
        $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_GROUPS_UPDATED', $title), RESPONSE_NOTICE);

        return true;
    }

    /**
     * Add a group of banner (by they ids) to a certain group
     *
     * @access  public
     * @param   int     $gid  Group's ID
     * @param   array   $banners Array with banner id
     * @return  array   Response (notice or error)
     */
    function AddBannersToGroup($gid, $banners)
    {
        $AllBanners = $this->GetBanners(-1, -1);
        foreach ($AllBanners as $banner) {
            $rank = array_search($banner['id'], $banners);
            $rank = ($rank === false)? 0: $rank + 1;
            if (($banner['gid'] == $gid) || ($rank != 0)) {
                $this->UpdateBannerGroup($banner['id'], -1, $gid, $rank);
            }
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_GROUPS_UPDATED_BANNERS'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Adds an banner to a group
     *
     * @access  public
     * @param   int     $bid  Banner's ID
     * @param   int     $gid  Group's ID
     * @param   int     $new_gid  Group's ID
     * @return  boolean Returns true if banner was sucessfully added to the group, false if not
     */
    function UpdateBannerGroup($bid, $gid, $new_gid, $rank)
    {
        $new_gid = ($rank == 0)? 0 : $new_gid;
        if (($bid != -1) && ($gid != -1)) {
            $sql = '
                UPDATE [[banners]] SET
                    [gid]        = {new_gid},
                    [rank]       = {rank},
                    [updatetime] = {now}
                WHERE [[banners]].[id] = {bid} AND [[banners]].[gid] = {gid}';
        } elseif ($gid != -1) {
            $sql = '
                UPDATE [[banners]] SET
                    [gid]        = {new_gid},
                    [rank]       = {rank},
                    [updatetime] = {now}
                WHERE [[banners]].[gid] = {gid}';
        } elseif ($bid != -1) {
            $sql = '
                UPDATE [[banners]] SET
                    [gid]        = {new_gid},
                    [rank]       = {rank},
                    [updatetime] = {now}
                WHERE [id] = {bid}';
        } else {
            $sql = '
                UPDATE [[banners]] SET
                    [gid]        = {new_gid},
                    [rank]       = {rank},
                    [updatetime] = {now}';
        }

        MDB2::loadFile('Date');
        $date = $GLOBALS['app']->loadDate();
        $params = array();
        $params['bid']     = $bid;
        $params['gid']     = $gid;
        $params['new_gid'] = $new_gid;
        $params['rank']    = $rank;
        $params['now']     = MDB2_Date::mdbNow();

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        return true;
    }

    /**
     * Delete a banner
     *
     * @access  public
     * @param   int     $bid The banner that will be deleted
     * @return  boolean True if query was successful and Jaws_Error on error
     */
    function DeleteBanner($bid)
    {
        $banner = $this->GetBanner($bid);
        if (Jaws_Error::IsError($banner)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        if(!isset($banner['id'])) {
            $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_BANNERS_ERROR_DOES_NOT_EXISTS'), RESPONSE_ERROR);
            return false;
        }

        $sql = 'DELETE FROM [[banners]] WHERE [id] = {bid}';
        $res = $GLOBALS['db']->query($sql, array('bid' => $bid));
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_BANNERS_DELETED', $banner['title']), RESPONSE_NOTICE);
        if (!empty($banner['banner'])) {
            if (file_exists(JAWS_IPATH . $banner['banner'])) {
                @unlink(JAWS_IPATH . $banner['banner']);
            }
        }

        return true;
    }

    /**
     * Reset banner's views counter
     *
     * @access  public
     * @param   int     $bid   banner ID
     * @return  array   Response (notice or error)
     */
    function ResetViews($bid)
    {
        $banner = $this->GetBanner($bid);
        if (Jaws_Error::IsError($banner)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        if(!isset($banner['id'])) {
            $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_BANNERS_ERROR_DOES_NOT_EXISTS'), RESPONSE_ERROR);
            return false;
        }

        $sql = '
            UPDATE [[banners]] SET
                [views]      = {views},
                [updatetime] = {now}
            WHERE [[banners]].[id] = {bid}';

        MDB2::loadFile('Date');
        $date = $GLOBALS['app']->loadDate();
        $params = array();
        $params['bid']     = $bid;
        $params['views']   = 0;
        $params['now']     = MDB2_Date::mdbNow();
        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_BANNERS_UPDATED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Reset banner's clicks counter
     *
     * @access  public
     * @param   int     $bid   banner ID
     * @return  array   Response (notice or error)
     */
    function ResetClicks($bid)
    {
        $banner = $this->GetBanner($bid);
        if (Jaws_Error::IsError($banner)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        if(!isset($banner['id'])) {
            $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_BANNERS_ERROR_DOES_NOT_EXISTS'), RESPONSE_ERROR);
            return false;
        }

        $sql = '
            UPDATE [[banners]] SET
                [clicks]     = {clicks},
                [updatetime] = {now}
            WHERE [[banners]].[id] = {bid}';

        MDB2::loadFile('Date');
        $date = $GLOBALS['app']->loadDate();
        $params = array();
        $params['bid']     = $bid;
        $params['clicks']  = 0;
        $params['now']     = MDB2_Date::mdbNow();
        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_BANNERS_UPDATED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Delete a group
     *
     * @access  public
     * @param   int     $bid The banner that will be deleted
     * @return  boolean True if query was successful and Jaws_Error on error
     */
    function DeleteGroup($gid)
    {
        if ($gid == 1) {
            $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_GROUPS_ERROR_NOT_DELETABLE'), RESPONSE_ERROR);
            return false;
        }
        $group = $this->GetGroup($gid);
        if (Jaws_Error::IsError($group)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        if(!isset($group['id'])) {
            $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_GROUPS_ERROR_DOES_NOT_EXISTS'), RESPONSE_ERROR);
            return false;
        }

        $this->UpdateBannerGroup(-1, $gid, 0, 0);
        $sql = 'DELETE FROM [[banners_groups]] WHERE [id] = {gid}';
        $res = $GLOBALS['db']->query($sql, array('gid' => $gid));
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('BANNER_GROUPS_DELETED', $gid), RESPONSE_NOTICE);

        return true;
    }

}