<?php
/**
 * Contact admin model
 *
 * @category   GadgetModel
 * @package    Contact
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2006-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/gpl.html
 */
require_once JAWS_PATH . 'gadgets/Contact/Model.php';

class ContactAdminModel extends ContactModel
{
    /**
     * Installs the gadget
     *
     * @access  public
     * @return  boolean True on successful installation, Jaws_Error otherwise
     */
    function InstallGadget()
    {
        $result = $this->installSchema('schema.xml');
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        //registry keys.
        $GLOBALS['app']->Registry->NewKeyEx(array('/gadgets/Contact/use_antispam',      'true'),
                                            array('/gadgets/Contact/use_captcha',       'true'),
                                            array('/gadgets/Contact/email_format',      'html'),
                                            array('/gadgets/Contact/enable_attachment', 'false'),
                                            array('/gadgets/Contact/comment',           '')
                                            );
        return true;
    }

    /**
     * Uninstalls the gadget
     *
     * @access  public
     * @return  boolean  True on success and Jaws_Error otherwise
     */
    function UninstallGadget()
    {
        $tables = array('contacts',
                        'contacts_recipients');
        foreach ($tables as $table) {
            $result = $GLOBALS['db']->dropTable($table);
            if (Jaws_Error::IsError($result)) {
                $gName  = _t('CONTACT_NAME');
                $errMsg = _t('GLOBAL_ERROR_GADGET_NOT_UNINSTALLED', $gName);
                $GLOBALS['app']->Session->PushLastResponse($errMsg, RESPONSE_ERROR);
                return new Jaws_Error($errMsg, $gName);
            }
        }

        // Registry keys
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/Contact/use_antispam');
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/Contact/use_captcha');
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/Contact/email_format');
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/Contact/enable_attachment');
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/Contact/comment');

        return true;
    }

    /**
     * Update the gadget
     *
     * @access  public
     * @param   string  $old    Current version (in registry)
     * @param   string  $new    New version (in the $gadgetInfo file)
     * @return  boolean  Success/Failure (Jaws_Error)
     */
    function UpdateGadget($old, $new)
    {
        $result = $this->installSchema('schema.xml', '', "$old.xml");
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        // ACL keys
        $GLOBALS['app']->ACL->NewKey('/ACL/gadgets/Contact/EditSentMessage',  'true');

        // Registry keys.
        $send_html = $GLOBALS['app']->Registry->Get('/gadgets/Contact/send_html') == 'true';
        $GLOBALS['app']->Registry->NewKey('/gadgets/Contact/use_captcha', 'true');
        $GLOBALS['app']->Registry->NewKey('/gadgets/Contact/email_format', $send_html? 'html' : 'text');
        $GLOBALS['app']->Registry->NewKey('/gadgets/Contact/enable_attachment', 'false');
        $GLOBALS['app']->Registry->NewKey('/gadgets/Contact/comment', '');
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/Contact/send_html');

        return true;
    }

    /**
     * Get information of a Contact Reply
     *
     * @access  public
     * @param   int Contact ID
     * @return  array Array of Contact Reply Information or Jaws_Error on failure
     */
    function GetReply($id)
    {
        $sql = '
            SELECT
                [id], [name], [email], [recipient], [subject], [msg_txt], [reply], [reply_sent]
            FROM [[contacts]]
            WHERE [id] = {id}';

        $result = $GLOBALS['db']->queryRow($sql, array('id' => $id));
        if (Jaws_Error::IsError($result)) {
            return new Jaws_Error($result->getMessage(), 'SQL');
        }

        return $result;
    }

    /**
     * Get a list of the Contacts
     *
     * @access  public
     * @param   boolean
     * @param   boolean
     * @param   boolean
     * @return  array Array of Contacts or Jaws_Error on failure
     */
    function GetContacts($limit = false, $offset = null)
    {
        if (is_numeric($limit)) {
            $res = $GLOBALS['db']->setLimit($limit, $offset);
            if (Jaws_Error::IsError($res)) {
                return new Jaws_Error($res->getMessage(), 'SQL');
            }
        }
        $sql = '
            SELECT
                [id], [name], [email], [subject], [createtime]
            FROM [[contacts]]
            ORDER BY [id] DESC';

        $result = $GLOBALS['db']->queryAll($sql);
        if (Jaws_Error::IsError($result)) {
            return new Jaws_Error($result->getMessage(), 'SQL');
        }

        return $result;
    }

    /**
     * Update contact information
     *
     * @access  public
     * @param   integer $id         Contact ID
     * @param   string  $name       Name
     * @param   string  $email      Email address
     * @param   string  $$company
     * @param   string  $url
     * @param   string  $tel
     * @param   string  $fax
     * @param   string  $mobile
     * @param   string  $address
     * @param   integer $recipient  Recipient ID
     * @param   string  $subject    Subject of message
     * @param   string  $message    Message content
     * @return  boolean Success/Failure
     */
    function UpdateContact($id, $name, $email, $company, $url, $tel, $fax, $mobile, $address, $recipient, $subject, $message)
    {
        $sql = '
            UPDATE [[contacts]] SET
                [name]       = {name},
                [email]      = {email},
                [company]    = {company},
                [url]        = {url},
                [tel]        = {tel},
                [fax]        = {fax},
                [mobile]     = {mobile},
                [address]    = {address},
                [recipient]  = {recipient},
                [subject]    = {subject},
                [msg_txt]    = {message},
                [updatetime] = {now}
            WHERE [id] = {id}';

        MDB2::loadFile('Date');
        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $params              = array();
        $params['id']        = (int)$id;;
        $params['name']      = $xss->parse($name);
        $params['email']     = $xss->parse($email);
        $params['company']   = $xss->parse($company);
        $params['url']       = $xss->parse($url);
        $params['tel']       = $xss->parse($tel);
        $params['fax']       = $xss->parse($fax);
        $params['mobile']    = $xss->parse($mobile);
        $params['address']   = $xss->parse($address);
        $params['recipient'] = (int)$recipient;
        $params['subject']   = $xss->parse($subject);
        $params['message']   = $xss->parse($message);
        $params['now']       = MDB2_Date::mdbNow();

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse($result->GetMessage(), RESPONSE_ERROR);
            return new Jaws_Error($result->GetMessage(), _t('CONTACT_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_CONTACTS_UPDATED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Update contact reply
     *
     * @access  public
     * @param   integer $id     Contact ID
     * @param   string  $reply  Reply content
     * @return  boolean Success/Failure
     */
    function UpdateReply($id, $reply)
    {
        $sql = '
            UPDATE [[contacts]] SET
                [reply]      = {reply},
                [updatetime] = {now}
            WHERE [id] = {id}';

        MDB2::loadFile('Date');
        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $params          = array();
        $params['id']    = (int)$id;;
        $params['reply'] = $xss->parse($reply);
        $params['now']   = MDB2_Date::mdbNow();

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_ERROR_REPLY_NOT_UPDATED'), RESPONSE_ERROR);
            return new Jaws_Error($result->GetMessage(), _t('CONTACT_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_REPLY_UPDATED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Update reply send field
     *
     * @access  public
     * @param   integer $id Contact ID
     * @return  boolean Success/Failure
     */
    function UpdateReplySent($id, $reply_sent)
    {
        $sql = '
            UPDATE [[contacts]] SET
                [reply_sent] = {reply_sent},
                [updatetime] = {now}
            WHERE [id] = {id}';

        MDB2::loadFile('Date');
        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $params = array();
        $params['id']         = (int)$id;;
        $params['reply_sent'] = (int)$reply_sent;
        $params['now']        = MDB2_Date::mdbNow();

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_ERROR_REPLY_NOT_UPDATED'), RESPONSE_ERROR);
            return new Jaws_Error($result->GetMessage(), _t('CONTACT_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_REPLY_UPDATED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Send contact reply
     *
     * @access  public
     * @param   integer $cid Contact ID
     * @return  boolean Success/Failure
     */
    function SendReply($cid)
    {
        $contact = $this->GetReply($cid);
        if (Jaws_Error::IsError($contact)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('GLOBAL_ERROR_QUERY_FAILED'), _t('CONTACT_NAME'));
        }
        if (!isset($contact['id'])) {
            $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_ERROR_CONTACT_DOES_NOT_EXISTS'), RESPONSE_ERROR);
            return new Jaws_Error(_t('CONTACT_ERROR_CONTACT_DOES_NOT_EXISTS'), _t('CONTACT_NAME'));
        }
        $to = $contact['email'];
        $subject = _t('CONTACT_REPLY') . '[' . $contact['subject'] . ']';

        $rid = $contact['recipient'];
        if ($rid != 0) {
            $recipient = $this->GetRecipient($rid);
            if (Jaws_Error::IsError($recipient)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('GLOBAL_ERROR_QUERY_FAILED'), _t('CONTACT_NAME'));
            }
            if (!isset($recipient['id'])) {
                $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_ERROR_RECIPIENT_DOES_NOT_EXISTS'), RESPONSE_ERROR);
                return new Jaws_Error(_t('CONTACT_ERROR_RECIPIENT_DOES_NOT_EXISTS'), _t('CONTACT_NAME'));
            }
            $from_name  = $recipient['name'];
            $from_email = $recipient['email'];
        } else {
            $from_name  = $GLOBALS['app']->Registry->Get('/config/owner_name');
            $from_email = $GLOBALS['app']->Registry->Get('/config/owner_email');
        }

        $format = $GLOBALS['app']->Registry->Get('/gadgets/Contact/email_format');
        if ($format == 'html') {
            require_once JAWS_PATH . 'include/Jaws/StringHtml.php';
            $reply = Jaws_StringHtml::AutoParagraph($contact['reply']);
        } else {
            $reply = $contact['reply'];
        }

        $tpl = new Jaws_Template('gadgets/Contact/templates/');
        $tpl->Load('SendReplyTo.html', true);
        $tpl->SetBlock($format);
        $tpl->SetVariable('lbl_name',      _t('GLOBAL_NAME'));
        $tpl->SetVariable('lbl_email',     _t('GLOBAL_EMAIL'));
        $tpl->SetVariable('lbl_message',   _t('CONTACT_MESSAGE'));
        $tpl->SetVariable('lbl_reply',     _t('CONTACT_REPLY'));
        $tpl->SetVariable('name',          $contact['name']);
        $tpl->SetVariable('email',         $contact['email']);
        $tpl->SetVariable('message',       $contact['msg_txt']);
        $tpl->SetVariable('reply',         $reply);
        $tpl->ParseBlock($format);
        $template = $tpl->Get();

        require_once JAWS_PATH . 'include/Jaws/Mail.php';
        $mail = new Jaws_Mail;
        $mail->SetHeaders($to, $from_name, $from_email, $subject);
        $mail->AddRecipient($to, false, false);
        $mail->SetBody($template, $format);
        $result = $mail->send();
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_ERROR_REPLY_NOT_SENT'), RESPONSE_ERROR);
            return false;
        }

        $this->UpdateReplySent($cid, true);
        $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_REPLY_SENT'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Delete a Contact
     *
     * @access  public
     * @param   string $id  ID of the Contact
     * @return  boolean True on success and Jaws_Error on failure
     */
    function DeleteContact($id)
    {
        $sql = 'DELETE FROM [[contacts]] WHERE [id] = {id}';
        $result = $GLOBALS['db']->query($sql, array('id' => $id));
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_ERROR_CONTACT_NOT_DELETED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('CONTACT_ERROR_CONTACT_NOT_DELETED'), _t('CONTACT_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_CONTACTS_DELETED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Insert the information of a Recipient
     *
     * @access  public
     * @param   string $name    Name of the recipient
     * @param   string $email   Email of recipient
     * @param   string $tel     Phone number of recipient
     * @param   string $fax     Fax number of recipient
     * @param   string $mobile  Mobile number of recipient
     * @param   string $visible The visible of the recipient
     * @return  boolean True on success and Jaws_Error on failure
     */
    function InsertRecipient($name, $email, $tel, $fax, $mobile, $inform_type, $visible)
    {
        $sql = '
            INSERT INTO [[contacts_recipients]]
                ([name], [email], [tel], [fax], [mobile], [inform_type], [visible])
            VALUES
                ({name}, {email}, {tel}, {fax}, {mobile}, {inform_type}, {visible})';

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $params = array();
        $params['name']        = $xss->parse($name);
        $params['email']       = $xss->parse($email);
        $params['tel']         = $xss->parse($tel);
        $params['fax']         = $xss->parse($fax);
        $params['mobile']      = $xss->parse($mobile);
        $params['inform_type'] = (int)$inform_type;
        $params['visible']     = (int)$visible;

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_ERROR_RECIPIENT_NOT_ADDED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('CONTACT_ERROR_RECIPIENT_NOT_ADDED'),_t('CONTACT_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_RECIPIENT_ADDED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Update the information of a Recipient
     *
     * @access  public
     * @param   string $id      ID of the recipient
     * @param   string $name    Name of the recipient
     * @param   string $email   Email of recipient
     * @param   string $tel     Phone number of recipient
     * @param   string $fax     Fax number of recipient
     * @param   string $mobile  Mobile number of recipient
     * @param   string $visible The visible of the recipient
     * @return  boolean True on success and Jaws_Error on failure
     */
    function UpdateRecipient($id, $name, $email, $tel, $fax, $mobile, $inform_type, $visible)
    {
        $sql = '
            UPDATE [[contacts_recipients]] SET
                [name]        = {name},
                [email]       = {email},
                [tel]         = {tel},
                [fax]         = {fax},
                [mobile]      = {mobile},
                [inform_type] = {inform_type},
                [visible]     = {visible}
            WHERE [id] = {id}';

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $params = array();
        $params['id']          = (int)$id;
        $params['name']        = $xss->parse($name);
        $params['email']       = $xss->parse($email);
        $params['tel']         = $xss->parse($tel);
        $params['fax']         = $xss->parse($fax);
        $params['mobile']      = $xss->parse($mobile);
        $params['inform_type'] = (int)$inform_type;
        $params['visible']     = (int)$visible;

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_ERROR_RECIPIENT_NOT_UPDATED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('CONTACT_ERROR_RECIPIENT_NOT_UPDATED'), _t('CONTACT_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_RECIPIENT_UPDATED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Delete a Recipient
     *
     * @access  public
     * @param   string $id  ID of the Recipient
     * @return  boolean True on success and Jaws_Error on failure
     */
    function DeleteRecipient($id)
    {
        $sql = 'DELETE FROM [[contacts_recipients]] WHERE [id] = {id}';
        $result = $GLOBALS['db']->query($sql, array('id' => $id));
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_ERROR_RECIPIENT_NOT_DELETED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('CONTACT_ERROR_RECIPIENT_NOT_DELETED'), _t('CONTACT_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_RECIPIENT_DELETED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Set properties of the gadget
     *
     * @access  public
     * @param   boolean $use_antispam
     * @param   boolean $use_captcha
     * @param   string  $email_format
     * @return  boolean True if change is successful, if not, returns Jaws_Error on any error
     */
    function UpdateProperties($use_antispam, $use_captcha, $email_format)
    {
        $rs = array();
        $rs[] = $GLOBALS['app']->Registry->Set('/gadgets/Contact/use_antispam', $use_antispam);
        $rs[] = $GLOBALS['app']->Registry->Set('/gadgets/Contact/use_captcha',  $use_captcha);
        $rs[] = $GLOBALS['app']->Registry->Set('/gadgets/Contact/email_format', $email_format);

        foreach ($rs as $r) {
            if (Jaws_Error::IsError($r) || !$r) {
                $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_ERROR_RECIPIENT_NOT_UPDATED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('CONTACT_ERROR_RECIPIENT_NOT_UPDATED'), _t('CONTACT_NAME'));
            }
        }
        $GLOBALS['app']->Registry->Commit('Contact');
        $GLOBALS['app']->Session->PushLastResponse(_t('CONTACT_PROPERTIES_UPDATED'), RESPONSE_NOTICE);
        return true;
    }
}
