<?php
/**
 * Faq Admin Gadget
 *
 * @category   GadgetModel
 * @package    Faq
 * @author     Jonathan Hernandez <ion@suavizado.com>
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2004-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/gpl.html
 */
require_once JAWS_PATH . 'gadgets/Faq/Model.php';
class FaqAdminModel extends FaqModel
{
    /**
     * Installs the gadget
     *
     * @access       public
     * @return       true on successful installation, Jaws_Error otherwise
     */
    function InstallGadget()
    {
        $result = $this->installSchema('schema.xml');
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        MDB2::loadFile('Date');
        $variables = array();
        $variables['timestamp'] = MDB2_Date::mdbNow();

        $result = $this->installSchema('insert.xml', $variables, 'schema.xml', true);
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        return true;
    }

    /**
     * Uninstalls the gadget
     *
     * @access  public
     * @return  boolean  Success/Failure (Jaws_Error)
     */
    function UninstallGadget()
    {
        $tables = array('faq',
                        'faq_category');
        foreach ($tables as $table) {
            $result = $GLOBALS['db']->dropTable($table);
            if (Jaws_Error::IsError($result)) {
                $gName  = _t('FAQ_NAME');
                $errMsg = _t('GLOBAL_ERROR_GADGET_NOT_UNINSTALLED', $gName);
                $GLOBALS['app']->Session->PushLastResponse($errMsg, RESPONSE_ERROR);
                return new Jaws_Error($errMsg, $gName);
            }
        }

        return true;
    }

    /**
     * Update the gadget
     *
     * @access  public
     * @param   string  $old    Current version (in registry)
     * @param   string  $new    New version (in the $gadgetInfo file)
     * @return  boolean  Success/Failure (Jaws_Error)
     */
    function UpdateGadget($old, $new)
    {
        $result = $this->installSchema('schema.xml', '', "$old.xml");
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        // Registry keys.

        return true;
    }

    /**
     * Max question position
     *
     * @access public
     * @param int $category Category ID
     * @return Max position
     */
    function GetMaxQuestionPosition($category)
    {
        $sql = 'SELECT MAX([faq_position]) FROM [[faq]] WHERE [category] = {category}';
        $max = $GLOBALS['db']->queryOne($sql, array('category' => $category));
        if (Jaws_Error::IsError($max)) {
            $max = 0;
        }

        return $max;
    }

    /**
     * Add a new Question
     *
     * @access  public
     * @param   string  $question The question
     * @param   string  $fast_url Fast URL
     * @param   string  $answer   The answer of the question
     * @param   int     $category Category id
     * @param   boolean $active   Question status
     * @return  boolean True if question is succesfully added, Jaws_Error if not
     */
    function AddQuestion($question, $fast_url, $answer, $category, $active)
    {
        $fast_url = empty($fast_url) ? $question : $fast_url;
        $fast_url = $this->GetRealFastUrl($fast_url, 'faq');

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        MDB2::loadFile('Date');
        $params             = array();
        $params['question'] = $xss->parse($question);
        $params['fast_url'] = $xss->parse($fast_url);
        $params['answer']   = $xss->parse($answer, false);
        $params['category'] = $category;
        $params['active']   = $active;
        $params['position'] = $this->GetMaxQuestionPosition($category) + 1;
        $params['now']      = MDB2_Date::mdbNow();

        $sql = '
            INSERT INTO [[faq]]
                ([question], [fast_url], [answer], [category], [published], [faq_position], [createtime], [updatetime])
            VALUES
                ({question}, {fast_url}, {answer}, {category}, {active}, {position}, {now}, {now})';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_QUESTION_NOT_ADDED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_QUESTION_NOT_ADDED'), _t('FAQ_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_QUESTION_ADDED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Update a question
     *
     * @access  public
     * @param   string  $id       Number of the question
     * @param   string  $question The question
     * @param   string  $fast_url Fast URL
     * @param   string  $answer   The answer of the question
     * @param   int     $category Category id
     * @param   boolean $active   Question status
     * @return  boolean True if question is succesfully updated, Jaws_Error if not
     */
    function UpdateQuestion($id, $question, $fast_url, $answer, $category, $active)
    {
        $fast_url = empty($fast_url) ? $question : $fast_url;
        $fast_url = $this->GetRealFastUrl($fast_url, 'faq', false);

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        MDB2::loadFile('Date');
        $params             = array();
        $params['id']       = $id;
        $params['question'] = $xss->parse($question);
        $params['fast_url'] = $xss->parse($fast_url);
        $params['answer']   = $xss->parse($answer, false);
        $params['category'] = $category;
        $params['active']   = $active;
        $params['now']      = MDB2_Date::mdbNow();

        $sql = '
            UPDATE [[faq]] SET
                [question]   = {question},
                [fast_url]   = {fast_url},
                [answer]     = {answer},
                [category]   = {category},
                [published]  = {active},
                [updatetime] = {now}
            WHERE [id] = {id}';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_QUESTION_NOT_UPDATED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_QUESTION_NOT_UPDATED'), _t('FAQ_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_QUESTION_UPDATED'), RESPONSE_NOTICE);
        return true;
    }


    /**
     * Delete a question
     *
     * @access  public
     * @param   string  $id   Number of the question
     * @return  boolean True if question is succesfully deleted, Jaws_Error if not
     */
    function DeleteQuestion($id)
    {
        $sql = 'SELECT [faq_position] FROM [[faq]] WHERE [id] = {id}';
        $rid = $GLOBALS['db']->queryRow($sql, array('id' => $id));
        if (Jaws_Error::IsError($rid)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_QUESTION_NOT_DELETED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_QUESTION_NOT_DELETED'), _t('FAQ_NAME'));
        }

        if (isset($rid['faq_position'])) {
            $sql = '
                UPDATE [[faq]] SET
                    [faq_position] = [faq_position] - 1
                WHERE [faq_position] > {pos}';

            $rs = $GLOBALS['db']->query($sql, array('pos' => $rid['faq_position']));
            if (Jaws_Error::IsError($rs)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_QUESTION_NOT_DELETED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('FAQ_ERROR_QUESTION_NOT_DELETED'), _t('FAQ_NAME'));
            }

            $sql = 'DELETE FROM [[faq]] WHERE [id] = {id}';
            $rs = $GLOBALS['db']->query($sql, array('id' => $id));
            if (Jaws_Error::IsError($rs)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_QUESTION_NOT_DELETED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('FAQ_ERROR_QUESTION_NOT_DELETED'), _t('FAQ_NAME'));
            }

            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_QUESTION_DELETED'), RESPONSE_NOTICE);
            return true;
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_QUESTION_DOES_NOT_EXISTS'), RESPONSE_ERROR);
        return new Jaws_Error(_t('FAQ_ERROR_QUESTION_DOES_NOT_EXISTS'));
    }

    /**
     * Move a given question
     *
     * @access  public
     * @param   string  $direction Where to move it
     * @param   int     $id  Question id
     * @return  boolean Returns true if the question was moved without problems, if not, returns Jaws_Error
     */
    function MoveQuestion($direction, $id)
    {
        $item = $this->getQuestion($id);
        $sql = '
            SELECT
                [id], [faq_position]
            FROM [[faq]]
            WHERE [category] = {category}
            ORDER BY [faq_position] ASC';

        $rs = $GLOBALS['db']->queryAll($sql, array('category' => $item['category_id']));
        if (Jaws_Error::IsError($rs)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_QUESTION_NOT_MOVED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_QUESTION_NOT_MOVED'), _t('FAQ_NAME'));
        }

        $qarray = array();
        foreach($rs as $row) {
            $res['id'] = $row['id'];
            $res['position'] = $row['faq_position'];
            $qarray[$row['id']] = $res;
        }
        reset($qarray);

        if (!is_array($qarray) || (is_array($qarray) && count($qarray) == 0)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_QUESTION_NOT_MOVED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_QUESTION_NOT_MOVED'), _t('FAQ_NAME'));
        }

        $found = false;
        while (!$found) {
            $v = current($qarray);
            if ($v['id'] == $id) {
                $found = true;
                $position = $v['position'];
                $id = $v['id'];
            } else {
                next($qarray);
            }
        }
        $run_queries = false;

        if ($direction == 'UP' && prev($qarray)) {
            $v = current($qarray);
            $m_position = $v['position'];
            $m_id = $v['id'];
            $run_queries = true;
        }

        if ($direction == 'DOWN' && next($qarray)) {
            $v = current($qarray);
            $m_position = $v['position'];
            $m_id = $v['id'];
            $run_queries = true;
        }

        if ($run_queries) {
            $params = array();
            $params['position'] = $m_position;
            $params['id'] = $id;

            $sql = '
                UPDATE [[faq]] SET
                    [faq_position] = {position}
                WHERE [id] = {id}';

            $result = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($result)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_QUESTION_NOT_MOVED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('FAQ_ERROR_QUESTION_NOT_MOVED'), _t('FAQ_NAME'));
            }

            $params = array();
            $params['position'] = $position;
            $params['id'] = $m_id;

            $sql = '
                UPDATE [[faq]] SET
                    [faq_position] = {position}
                WHERE [id] = {id}';

            $result = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($result)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_QUESTION_NOT_MOVED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('FAQ_ERROR_QUESTION_NOT_MOVED'), _t('FAQ_NAME'));
            }
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_QUESTION_MOVED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Max category position
     *
     * @access public
     * @return Max position
     */
    function GetMaxCategoryPosition()
    {
        $sql = 'SELECT MAX([category_position]) FROM [[faq_category]]';
        $max = $GLOBALS['db']->queryOne($sql);
        if (Jaws_Error::IsError($max)) {
            $max = 0;
        }

        return $max;
    }

    /**
     * Add a category
     *
     * @access  public
     * @param   string  $category     Category name
     * @param   string  $fast_url     Fast URL
     * @param   string  $description  Category description
     * @return  boolean True if success, Jaws_Error otherwise
     */
    function AddCategory($category, $fast_url, $description)
    {
        $fast_url = empty($fast_url) ? $category : $fast_url;
        $fast_url = $this->GetRealFastUrl($fast_url, 'faq_category');

        MDB2::loadFile('Date');
        $params                = array();
        $params['category']    = $category;
        $params['fast_url']    = $fast_url;
        $params['description'] = $description;
        $params['position']    = $this->GetMaxCategoryPosition() + 1;
        $params['updatetime']  = MDB2_Date::mdbNow();

        $sql = '
            INSERT INTO [[faq_category]]
                ([category], [fast_url], [description], [category_position], [updatetime])
            VALUES
                ({category}, {fast_url}, {description}, {position}, {updatetime})';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_ADDED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_ADDED'), _t('FAQ_NAME'));
        }

        $cid = $GLOBALS['db']->lastInsertID('faq_category', 'id');
        if (Jaws_Error::IsError($cid) || empty($cid)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_ADDED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_ADDED'), _t('FAQ_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_CATEGORY_ADDED'), RESPONSE_NOTICE);
        return $cid;
    }

    /**
     * Update a category
     *
     * @access  public
     * @param   int     $id           Category ID
     * @param   string  $category     Category name
     * @param   string  $fast_url     Fast URL
     * @param   string  $description  Category description
     * @return  boolean True if category is succesfully updated, Jaws_Error if not
     */
    function UpdateCategory($id, $category, $fast_url, $description)
    {
        $fast_url = empty($fast_url) ? $category : $fast_url;
        $fast_url = $this->GetRealFastUrl($fast_url, 'faq_category', false);

        MDB2::loadFile('Date');
        $params = array();
        $params['id'] = $id;
        $params['category']    = $category;
        $params['fast_url']    = $fast_url;
        $params['description'] = $description;
        $params['updatetime']  = MDB2_Date::mdbNow();

        $sql = '
            UPDATE [[faq_category]] SET
                [category] = {category},
                [fast_url] = {fast_url},
                [description] = {description},
                [updatetime] = {updatetime}
            WHERE [id] = {id}';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_UPDATED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_UPDATED'), _t('FAQ_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_CATEGORY_UPDATED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Delete category
     *
     * @access public
     * @param int $id position
     * @return True if success, Jaws_Error on failure
     */
    function DeleteCategory($id)
    {
        $sql = 'SELECT [category_position] FROM [[faq_category]] WHERE [id] = {id}';
        $row = $GLOBALS['db']->queryRow($sql, array('id' => $id));
        if (Jaws_Error::IsError($row)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_DELETED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_UPDATED'), _t('FAQ_NAME'));
        }

        if (isset($row['category_position'])) {
            $sql = '
                UPDATE [[faq_category]] SET
                    [category_position] = [category_position] - 1
                WHERE [category_position] > {pos}';

            $result = $GLOBALS['db']->query($sql, array('pos' => $row['category_position']));
            if (Jaws_Error::IsError($result)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_DELETED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_UPDATED'), _t('FAQ_NAME'));
            }

            $sql = 'DELETE FROM [[faq_category]] WHERE [id] = {id}';
            $result = $GLOBALS['db']->query($sql, array('id' => $id));
            if (Jaws_Error::IsError($result)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_DELETED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_DELETED'), _t('FAQ_NAME'));
            }

            $sql = 'DELETE FROM [[faq]] WHERE [category] = {id}';
            $result = $GLOBALS['db']->query($sql, array('id' => $id));
            if (Jaws_Error::IsError($result)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_DELETED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_DELETED'), _t('FAQ_NAME'));
            }

            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_CATEGORY_DELETED'), RESPONSE_NOTICE);
            return true;
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_DOES_NOT_EXISTS'), RESPONSE_ERROR);
        return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_DOES_NOT_EXISTS'));
    }

    /**
     * Fix the position of a category
     *
     * @access  public
     * @param   int    $cat  Category ID
     * @param   int    $pos  New position
     * @return  Returns true if the category was moved without problems, if not, returns Jaws_Error
     */
    function FixCategoryPosition($cat, $pos)
    {
        $params = array();
        $params['position'] = $pos;
        $params['id']       = $cat;
        $sql = '
            UPDATE [[faq_category]] SET
                [category_position] = {position}
            WHERE [id] = {id}';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_MOVED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_MOVED'), _t('FAQ_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_CATEGORY_MOVED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Move a given category
     *
     * @access  public
     * @param   string  $direction Where to move it
     * @param   int     $id  category id
     * @return  boolean Returns true if the category was moved without problems, if not, returns Jaws_Error
     */
    function MoveCategory($direction, $id)
    {
        $sql = '
            SELECT
                [id], [category_position]
            FROM [[faq_category]]
            ORDER BY [category_position] ASC';

        $result = $GLOBALS['db']->queryAll($sql);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_MOVED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_MOVED'), _t('FAQ_NAME'));
        }

        $qarray = array();
        foreach ($result as $row) {
            $res['id'] = $row['id'];
            $res['position'] = $row['category_position'];
            $qarray[$row['id']] = $res;
        }
        reset($qarray);

        if ((!is_array($qarray)) ||((is_array($qarray)) &&(count($qarray) == 0))) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_MOVED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_MOVED'), _t('FAQ_NAME'));
        }

        $found = false;
        while (!$found) {
            $v = current($qarray);
            if ($v['id'] == $id) {
                $found = true;
                $position = $v['position'];
                $id = $v['id'];
            } else {
                next($qarray);
            }
        }
        $run_queries = false;

        if ($direction == 'UP' && prev($qarray)) {
            $v = current($qarray);
            $m_position = $v['position'];
            $m_id = $v['id'];
            $run_queries = true;
        }

        if ($direction == 'DOWN' && next($qarray)) {
            $v = current($qarray);
            $m_position = $v['position'];
            $m_id = $v['id'];
            $run_queries = true;
        }

        if ($run_queries) {
            $params = array();
            $params['position'] = $m_position;
            $params['id']       = $id;
            $sql = '
                UPDATE [[faq_category]] SET
                    [category_position] = {position}
                WHERE [id] = {id}';

            $result = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($result)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_MOVED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_MOVED'), _t('FAQ_NAME'));
            }

            $params = array();
            $params['position'] = $position;
            $params['id']       = $m_id;
            $sql = '
                UPDATE [[faq_category]] SET
                    [category_position] = {position}
                WHERE [id] = {id}';

            $result = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($result)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_ERROR_CATEGORY_NOT_MOVED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('FAQ_ERROR_CATEGORY_NOT_MOVED'), _t('FAQ_NAME'));
            }
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('FAQ_CATEGORY_MOVED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Implements the GetRealFastURL
     *
     * @access  protected
     * @param   string     $fast_url     Fast URL
     * @param   string     $table        DB table name (with no [[ ]])
     * @param   boolean    $unique_check must be false in update methods
     * @param   string     $field        Table field where fast_url is stored
     * @return  string     Correct fast URL
     */
    function GetRealFastURL($fast_url, $table, $unique_check = true, $field = 'fast_url')
    {
        if (is_numeric($fast_url)) {
            $fast_url = '-' . $fast_url . '-';
        }
        $fast_url = $GLOBALS['app']->UTF8->str_replace(' ',  '-', $fast_url);
        $fast_url = $GLOBALS['app']->UTF8->str_replace('/',  '-', $fast_url);
        $fast_url = $GLOBALS['app']->UTF8->str_replace('\\', '-', $fast_url);

        if (!$unique_check) {
            return $fast_url;
        }

        $params = array();
        $params['fast_url'] = $fast_url;

        $sql = "
             SELECT COUNT(*)
             FROM [[$table]]
             WHERE [$field] = {fast_url}";

        $total = $GLOBALS['db']->queryOne($sql, $params);
        if (Jaws_Error::isError($total) || ($total == '0')) {
            return $fast_url;
        }

        //Get a list of fast_url's that match the current entry
        $params['fast_url'] = $GLOBALS['app']->UTF8->trim($fast_url).'%';

        $sql = "
             SELECT [$field]
             FROM [[$table]]
             WHERE [$field] LIKE {fast_url}";

        $urlList = $GLOBALS['db']->queryAll($sql, $params);
        if (Jaws_Error::IsError($urlList)) {
            return $fast_url;
        }

        $counter = 1;
        $numbers = array();
        foreach($urlList as $url) {
            //Matches the foo-\d?
            if (preg_match("/(.+?)-([0-9]{0,})/", $url[$field], $matches)) {
                $numbers[] = (int)$matches[2];
            }
        }
        if (count($numbers) == 0) {
            return $fast_url . '-1';
        }
        $lastNumber = $numbers[count($numbers)-1];
        return $fast_url.'-'.($lastNumber+1);
    }

}