<?php
/**
 * Filebrowser Admin Gadget
 *
 * @category   GadgetModel
 * @package    FileBrowser
 * @author     Jonathan Hernandez <ion@suavizado.com>
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2004-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/gpl.html
 */
require_once JAWS_PATH . 'gadgets/FileBrowser/Model.php';

class FileBrowserAdminModel extends FileBrowserModel
{
    /**
     * Install the gadget
     *
     * @access  public
     * @return  boolean True on successful installation, Jaws_Error otherwise
     */
    function InstallGadget()
    {
        if (!Jaws_Utils::is_writable(JAWS_DATA)) {
            return new Jaws_Error(_t('GLOBAL_ERROR_FAILED_DIRECTORY_UNWRITABLE', JAWS_DATA));
        }

        $new_dir = JAWS_DATA . 'files' . DIRECTORY_SEPARATOR;
        if (!Jaws_Utils::mkdir($new_dir)) {
            return new Jaws_Error(_t('GLOBAL_ERROR_FAILED_CREATING_DIR', $new_dir), _t('FILEBROWSER_NAME'));
        }

        //registry keys.
        $GLOBALS['app']->Registry->NewKey('/gadgets/FileBrowser/black_list', '.htaccess');
        $GLOBALS['app']->Registry->NewKey('/gadgets/FileBrowser/root_dir', 'files');
        $GLOBALS['app']->Registry->NewKey('/gadgets/FileBrowser/frontend_avail', 'true');

        return true;
    }

    /**
     * Uninstalls the gadget
     *
     * @access  public
     * @return  boolean  Success/Failure (Jaws_Error)
     */
    function UninstallGadget()
    {
        //registry keys.
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/FileBrowser/black_list');
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/FileBrowser/root_dir');
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/FileBrowser/frontend_avail');

        return true;
    }

    /**
     * Update the gadget
     *
     * @access  public
     * @param   string  $old    Current version (in registry)
     * @param   string  $new    New version (in the $gadgetInfo file)
     * @return  boolean  Success/Failure (Jaws_Error)
     */
    function UpdateGadget($old, $new)
    {
        if (version_compare($old, '0.7.0', '<')) {
            $result = $this->installSchema('schema.xml', '', "$old.xml");
            if (Jaws_Error::IsError($result)) {
                return $result;
            }

            $result = $GLOBALS['db']->dropTable('filebrowser_communities');
            if (Jaws_Error::IsError($result)) {
                // do nothing
            }

            // Registry keys.
            $GLOBALS['app']->Registry->NewKey('/gadgets/FileBrowser/black_list', '.htaccess');
            $GLOBALS['app']->Registry->NewKey('/gadgets/FileBrowser/frontend_avail', 'true');
        }

        // Registry keys.
        $GLOBALS['app']->Registry->NewKey('/gadgets/FileBrowser/root_dir', 'files');

        return true;
    }

    /**
     * Creates a directory
     *
     * @access  public
     * @param   string  $path     Where to create it
     * @param   string  $dir_name Which name
     * @return  boolean Returns true if the directory was created, if not, returns Jaws_Error
     */
    function CreateDir($path, $dir_name)
    {
        $dir = $this->GetFileBrowserRootDir() . $path . '/' . $dir_name;

        require_once 'File/Util.php';
        $realpath = File_Util::realpath($dir);
        $blackList = explode(',', $GLOBALS['app']->Registry->Get('/gadgets/FileBrowser/black_list'));
        $blackList = array_map('strtolower', $blackList);

        if (!File_Util::pathInRoot($realpath, $this->GetFileBrowserRootDir()) ||
            in_array(strtolower(basename($realpath)), $blackList) ||
            !Jaws_Utils::mkdir($realpath))
        {
            $GLOBALS['app']->Session->PushLastResponse(_t('FILEBROWSER_ERROR_CANT_CREATE_DIRECTORY', $realpath), RESPONSE_ERROR);
            return new Jaws_Error(_t('FILEBROWSER_ERROR_CANT_CREATE_DIRECTORY', $realpath), _t('FILEBROWSER_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('FILEBROWSER_CREATED_DIR', $dir_name), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Deletes a file or directory
     *
     * @access  public
     * @param   string  $path     Where is it
     * @param   string  $filename The name of the file
     * @return  boolean Returns true if file/directory was deleted without problems, if not, returns Jaws_Error
     */
    function DeleteFile($path, $filename)
    {
        if ($path[0]=='\\' || $path[0]=='/') $path = substr($path, 1);
        if ($filename[0]=='\\' || $filename[0]=='/') $filename = substr($filename, 1);
        $file = $path . ((empty($path)? '': DIRECTORY_SEPARATOR)) . $filename;
        $filename = $this->GetFileBrowserRootDir() . DIRECTORY_SEPARATOR . $file;
        $blackList = explode(',', $GLOBALS['app']->Registry->Get('/gadgets/FileBrowser/black_list'));
        $blackList = array_map('strtolower', $blackList);

        require_once 'File/Util.php';
        $realpath = File_Util::realpath($filename);
        if (!File_Util::pathInRoot($realpath, $this->GetFileBrowserRootDir()) ||
            in_array(strtolower(basename($filename)), $blackList))
        {
            $msgError = is_dir($filename)? _t('FILEBROWSER_ERROR_CANT_DELETE_DIR', $file):
                                           _t('FILEBROWSER_ERROR_CANT_DELETE_FILE', $file);
            $GLOBALS['app']->Session->PushLastResponse($msgError, RESPONSE_ERROR);
            return new Jaws_Error($msgError, _t('FILEBROWSER_NAME'));
        }

        if (is_file($filename)) {
            $return = @unlink($filename);

            if ($return) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FILEBROWSER_DELETED_FILE', $file), RESPONSE_NOTICE);
                return true;
            }

            $GLOBALS['app']->Session->PushLastResponse(_t('FILEBROWSER_ERROR_CANT_DELETE_FILE', $file), RESPONSE_ERROR);
            return new Jaws_Error(_t('FILEBROWSER_ERROR_CANT_DELETE_FILE', $file), _t('FILEBROWSER_NAME'));
        }

        if (is_dir($filename)) {
            require_once JAWS_PATH . 'include/Jaws/FileManagement.php';
            $return = Jaws_FileManagement::FullRemoval($filename);

            if ($return) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FILEBROWSER_DELETED_DIR', $file), RESPONSE_NOTICE);
                return true;
            }

            $GLOBALS['app']->Session->PushLastResponse(_t('FILEBROWSER_ERROR_CANT_DELETE_DIR', $file), RESPONSE_ERROR);
            return new Jaws_Error(_t('FILEBROWSER_ERROR_CANT_DELETE_DIR', $file), _t('FILEBROWSER_NAME'));
        }
    }

    /**
     * Rename a given file or directory
     *
     * @access  public
     * @param   string  $type             file or dir
     * @param   string  $old_filename     Filename to rename
     * @param   string  $new_filename     New Filename
     * @return  boolean Returns file if file/directory was renamed without problems, if not, returns Jaws_Error
     */
    function Rename($type, $old, $new)
    {
        $type = strtoupper($type);
        $old = $this->GetFileBrowserRootDir() . $old;
        $new = $this->GetFileBrowserRootDir() . $new;

        require_once 'File/Util.php';
        $old = File_Util::realpath($old);
        $new = File_Util::realpath($new);
        $blackList = explode(',', $GLOBALS['app']->Registry->Get('/gadgets/FileBrowser/black_list'));
        $blackList = array_map('strtolower', $blackList);

        if (!File_Util::pathInRoot($old, $this->GetFileBrowserRootDir()) ||
            !File_Util::pathInRoot($new, $this->GetFileBrowserRootDir()) ||
            in_array(strtolower(basename($old)), $blackList) ||
            in_array(strtolower(basename($new)), $blackList))
        {
            $msgError = _t('FILEBROWSER_ERROR_CANT_RENAME_' . $type, $old, $new);
            $GLOBALS['app']->Session->PushLastResponse($msgError, RESPONSE_ERROR);
            return new Jaws_Error($msgError, _t('FILEBROWSER_NAME'));
        }

        $return = @rename($old, $new);
        if ($return) {
            $GLOBALS['app']->Session->PushLastResponse(_t('FILEBROWSER_RENAMED_' . $type, $old, $new), RESPONSE_NOTICE);
            return true;
        }

        $msgError = _t('FILEBROWSER_ERROR_CANT_RENAME_' . $type, $old, $new);
        $GLOBALS['app']->Session->PushLastResponse($msgError, RESPONSE_ERROR);
        return new Jaws_Error($msgError, _t('FILEBROWSER_NAME'));
    }
}