<?php
/**
 * Filebrowser Gadget
 *
 * @category   GadgetModel
 * @package    FileBrowser
 * @author     Jonathan Hernandez <ion@suavizado.com>
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @copyright  2004-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/gpl.html
 */
class FileBrowserModel extends Jaws_Model
{
    var $_Name = 'FileBrowser';

    /**
     * Convert bytes to a nice size format
     *
     * @access  public
     * @param   int     $size Size in Bytes
     * @return  string  The size with its prefix
     */
    function NiceSize($size)
    {
        $prefixes = array('bytes', 'Kb', 'Mb', 'Gb', 'Tb');
        $i = 0;
        while ($size >= 1024) {
            $size = $size/1024;
            $i++;
        }
        $size = round($size,2);

        return $size . ' ' . $prefixes[$i];
    }

    /**
     * Get root dir
     *
     * @access  public
     * @return  string  The root directory
     */
    function GetFileBrowserRootDir()
    {
        static $root_dir;
        if (!isset($root_dir)) {
            $root_dir = trim($GLOBALS['app']->Registry->Get('/gadgets/FileBrowser/root_dir'));
            $root_dir = JAWS_DATA . $root_dir;
            $root_dir = str_replace('..', '', $root_dir);

            require_once 'File/Util.php';
            $root_dir = File_Util::realpath($root_dir) . DIRECTORY_SEPARATOR;
            if (!File_Util::pathInRoot($root_dir, JAWS_DATA)) {
                Jaws_Error::Fatal(_t('FILEBROWSER_ERROR_DIRECTORY_DOES_NOT_EXISTS'), __FILE__, __LINE__);
            }
        }

        return $root_dir;
    }

    /**
     * Get files of the current root dir
     *
     * @access  public
     * @param   string  $current_dir Current directory
     * @return  array   A list of directories or files of a certain directory
     */
    function GetCurrentRootDir($current_dir)
    {
        if (!is_dir($this->GetFileBrowserRootDir() . $current_dir)) {
            return new Jaws_Error(_t('FILEBROWSER_ERROR_DIRECTORY_DOES_NOT_EXISTS'),
                                 _t('FILEBROWSER_NAME'));
        }

        if (trim($current_dir) != '') {
            $path = $current_dir;
            if (substr($path, -1) != '/') {
                $path .= '/';
            }
        } else {
            $path = '/';
        }

        $path = str_replace('..', '', $path);

        $tree = array();
        $tree['/'] = '/';

        if (!empty($path)) {
            $parent_path = substr(strrev($path), 1);
            if (strpos($parent_path, '/')) {
                $parent_path = strrev(substr($parent_path, strpos($parent_path, '/'), strlen($parent_path)));
            } else {
                $parent_path = '';
            }

            $vpath = '';
            foreach (explode('/', $path) as $k) {
                if ($k != '') {
                    $vpath .= '/'.$k;
                    $tree[$vpath] = $k;
                }
            }
        } else {
            $tree[] = $path;
        }

        return $tree;
    }

    /**
     * Gets the directory content
     *
     * @access  public
     * @param   string  $path Where to read
     * @return  array   A list of properties of files and directories of a certain path and Jaws_Error on failure
     */
    function ReadDir($path)
    {
        if (!empty($path) && $path != '/') {
            if (substr($path, -1) != '/') {
                $path .= '/';
            }
        } else {
            $path = '';
        }

        $path = str_replace('..', '', $path);

        $folder = $this->GetFileBrowserRootDir() . $path;
        if (!file_exists($folder) || !$adr = scandir($folder)) {
            if (isset($GLOBALS['app']->Session)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('FILEBROWSER_ERROR_CANT_OPEN_DIRECTORY', $path),
                                                      RESPONSE_ERROR);
            }
            return new Jaws_Error(_t('FILEBROWSER_ERROR_CANT_OPEN_DIRECTORY', $path),  _t('FILEBROWSER_NAME'));
        }

        $files = array();
        $file_counter = 0;
        $date_obj = $GLOBALS['app']->loadDate();
        $theme = $GLOBALS['app']->Registry->Get('/config/theme');
        $theme_image_path = JAWS_DATA . 'themes/'.$theme.'/FileBrowser/images/';
        $theme_image = $GLOBALS['app']->getDataURL() . 'themes/'.$theme.'/FileBrowser/images/';
        foreach ($adr as $file) {
            //we should return only 'visible' files, not hidden files
            if ($file{0} != '.') {
                $filepath = $this->GetFileBrowserRootDir() . $path . $file;
                if (is_dir($filepath)) {
                    //Set the filename
                    $files[$file_counter]['filename'] = $file;
                    //Set is_dir to true(tutru)
                    $files[$file_counter]['is_dir'] = true;
                    //Set relative path
                    if (empty($path)) {
                        $files[$file_counter]['relative'] = str_replace('//', '/', $file);
                        $url = $GLOBALS['app']->Map->GetURLFor('FileBrowser', 'Display', array('path' => $file));
                    } else {
                        $files[$file_counter]['relative'] = str_replace('//', '/', $path.'/'.$file);
                        $url = $GLOBALS['app']->Map->GetURLFor('FileBrowser', 'Display',
                                                               array('path' => str_replace('//', '/', $path.'/'.$file)));
                    }
                    $files[$file_counter]['url'] = $url;
                    //Set the size(default for dirs)
                    $files[$file_counter]['size'] = $this->NiceSize(4096);
                    //Get $date
                    $date = $date_obj->ToISO(date('Y-m-d H:i:s', filemtime($filepath)));
                    //Set the date
                    $files[$file_counter]['date'] = $date;
                    //Set the curr dir name
                    $files[$file_counter]['dirname'] = $path;
                    //Fullpath
                    $files[$file_counter]['fullpath'] = $filepath;
                    //Is shared?
                    $files[$file_counter]['is_shared'] = file_exists($files[$file_counter]['fullpath'].'/.jaws_virtual');
                    //hold.. is it shared?
                    if ($files[$file_counter]['is_shared']) {
                        $files[$file_counter]['icon'] = 'gadgets/FileBrowser/images/folder-remote.png';
                        $files[$file_counter]['mini_icon'] =  'gadgets/FileBrowser/images/folder-remote.png';
                    } else {
                        $files[$file_counter]['icon'] = 'gadgets/FileBrowser/images/folder.png';
                        $files[$file_counter]['mini_icon'] = 'gadgets/FileBrowser/images/mini_folder.png';
                    }
                } else {
                    //Set the filename
                    $files[$file_counter]['filename'] = $file;
                    //Set is_dir to false
                    $files[$file_counter]['is_dir'] = false;
                    //Get url
                    if ($path == '/') $path = '';
                    $root_dir = trim($GLOBALS['app']->Registry->Get('/gadgets/FileBrowser/root_dir'));
                    $url = $GLOBALS['app']->getDataURL(true) . str_replace('//', '/', $root_dir.'/'.$path.'/'.$file);
                    //Set the url of the file
                    $files[$file_counter]['url'] = $url;
                    //Set relative path
                    $files[$file_counter]['relative'] = str_replace('//', '/', $path.'/'.$file);

                    //Get the extension
                    //require 'MIME/Type.php';
                    //var_dump(MIME_Type::autoDetect($filepath));
                    $file_extension = strtolower(strrev(substr(strrev($file), 0, strpos(strrev($file), '.'))));

                    //Get the icon
                    $iconName = $this->getExtImage($file_extension);
                    if (is_file($theme_image_path . $iconName)) {
                        $files[$file_counter]['icon'] = $theme_image . $iconName;
                    } else {
                        //Is icon does not exists..
                        if (!is_file(JAWS_PATH . 'gadgets/FileBrowser/images/'.$iconName)) {
                            $icon =  'gadgets/FileBrowser/images/mimetypes/text-x-generic.png';
                        } else {
                            $icon =  'gadgets/FileBrowser/images/'.$iconName;
                        }
                        $files[$file_counter]['icon'] = $icon;
                    }

                    //Set the extension
                    $files[$file_counter]['ext'] = $file_extension;
                    //Fullpath
                    $files[$file_counter]['fullpath'] = $filepath;
                    //Set the icon
                    $files[$file_counter]['mini_icon'] = 'gadgets/FileBrowser/images/mini_file.png';
                    //Get $date
                    $date = $date_obj->ToISO(date('Y-m-d H:i:s', filemtime($filepath)));
                    //Set the date
                    $files[$file_counter]['date'] = $date;
                    //Get the file size
                    $file_size = filesize($filepath);
                    //Set the file size
                    $files[$file_counter]['size'] = $this->NiceSize($file_size);
                    //Set the curr dir name
                    $files[$file_counter]['dirname'] = $path;
                }
                //Is it a VirtualDir?
                $file_counter ++;
            }
        }

        if ($files) {
            return $this->SortFiles($files);
        }

        return array();
    }

    /**
     * Performs case insensitive sort based on filename.
     * Directories first, followed by files.
     *
     * @access public
     * @param array $contents The filesystem array
     * @return array the sorted filesystem array
     */
    function SortFiles($contents)
    {
        require_once JAWS_PATH.'include/Jaws/ArraySort.php';
        $contents = Jaws_ArraySort::SortBySecondIndex($contents, 'is_dir');

        while (list($position, ) = each($contents)) {
            $isDir[$position] = $contents[$position]['is_dir'];
        }
        $dirStart = array_search('1', $isDir);
        $files = array_splice($contents, 0, $dirStart);

        if (!empty($files)) {
            $files    = Jaws_ArraySort::SortBySecondIndex($files, 'filename', false);
        }

        if (!empty($contents)) {
            $contents = Jaws_ArraySort::SortBySecondIndex($contents, 'filename', false);
        }

        return array_merge($contents, $files);
    }

    function getExtImage($ext)
    {
        $ext = strtolower($ext);

        $exts = array();
        $exts['audio-x-generic'] = array(
            'mp3', 'wav', 'aac', 'flac',
            'ogg', 'wma', 'cda', 'voc', 'midi',
            'ac3', 'bonk', 'mod'
        );

        $exts['image-x-generic'] = array(
            'gif', 'png', 'jpg', 'jpeg', 'raw',
            'bmp', 'tiff', 'swf', 'svg'
        );

        $exts['package-x-generic'] = array(
            'tar', 'tar.gz', 'zip', 'gzip', 'rar',
            'rpm', 'deb', 'iso', 'bz2', 'bak', 'gz'
        );

        $exts['video-x-generic'] = array(
            'mpg', 'mpeg', 'avi', 'wma', 'rm',
            'asf', 'flv', 'mov'
        );

        $exts['text-x-generic'] = array(
            'txt', 'pdf', ''
        );

        $exts['text-html'] = array(
            'html', 'htm'
        );

        $exts['text-x-script'] = array(
            'sh', 'pl', 'php', 'asp', 'jsp',
            'py', 'c', 'css'
        );

        $exts['application-x-executable'] = array(
            'exe'
        );

        $exts['x-office-document-template'] = array(
            'stw', 'ott',
        );

        $exts['x-office-document'] = array(
            'doc', 'sxw', 'odt', 'rtf', 'sdw'
        );

        $exts['x-office-presentation-template'] = array(
            'pot', 'otp', 'sti'
        );

        $exts['x-office-presentation'] = array(
            'ppt', 'odp', 'sxi'
        );

        $exts['x-office-spreadsheet-template'] = array(
            'xlt', 'ots', 'stc'
        );

        $exts['x-office-spreadsheet'] = array(
            'xls', 'ods', 'sxc', 'sdc'
        );

        $exts['x-office-drawing-template'] = array(
        );

        $exts['x-office-drawing'] = array(
            'sxd', 'sda', 'sdd', 'odg'
        );

        $found = false;
        foreach ($exts as $key => $data) {
            if (in_array($ext, $data)) {
                return 'mimetypes/' . $key . '.png';
            }
        }


        return false;
    }
}