<?php
/**
 * Languages Core Gadget
 *
 * @category   GadgetModel
 * @package    Languages
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2007-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
define('EMPTY_STRING', "-EMPTY-");

class LanguagesAdminModel extends Jaws_Model
{
    var $_Name = 'Languages';

    /**
     * Installs the gadget
     *
     * @access       public
     * @return       true on successful installation, Jaws_Error otherwise
     */
    function InstallGadget()
    {
        $GLOBALS['app']->Registry->NewKey('/gadgets/Languages/base_lang', 'en');
        $GLOBALS['app']->Registry->NewKey('/gadgets/Languages/pluggable', 'false');
        return true;
    }

    /**
     * @access  public
     *
     * @param   string  $lang_str   Language code and name
     * @return  boolean Success/Failure (Jaws_Error)
     */
    function SaveLanguage($lang_str)
    {
        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        if ($lang_str == $xss->parse($lang_str)) {
            $lang_code = substr($lang_str, 0, strpos($lang_str, ';'));
            if (preg_match("/^([a-z]{2})$|^([a-z]{2}[-][a-z]{2})$/", $lang_code)) {
                $lang_name = substr($lang_str, strpos($lang_str, ';')+1);
                if (!empty($lang_name) || trim($lang_name) == $lang_name) {
                    $jaws_lang_dir = JAWS_PATH. "languages";
                    $lang_dir = $jaws_lang_dir. DIRECTORY_SEPARATOR. $lang_code;

                    $lang_exist = @is_dir($lang_dir);
                    if (!Jaws_Utils::is_writable($jaws_lang_dir)) {
                        $GLOBALS['app']->Session->PushLastResponse(
                                            _t('GLOBAL_ERROR_FAILED_DIRECTORY_UNWRITABLE'),
                                            RESPONSE_ERROR);
                        return false;
                    }
                    if (!Jaws_Utils::mkdir($lang_dir)) {
                        $GLOBALS['app']->Session->PushLastResponse(
                                            _t('GLOBAL_ERROR_FAILED_CREATING_DIR'),
                                            RESPONSE_ERROR);
                        return false;
                    }

                    $lang_fname_file = $lang_dir. DIRECTORY_SEPARATOR. 'FullName';
                    if (Jaws_Utils::file_put_contents($lang_fname_file, $lang_name)) {
                        if ($lang_exist) {
                            $GLOBALS['app']->Session->PushLastResponse(
                                            _t('LANGUAGES_LANGUAGE_UPDATED', $lang_code),
                                            RESPONSE_NOTICE);
                        } else {
                            $GLOBALS['app']->Session->PushLastResponse(
                                            _t('LANGUAGES_LANGUAGE_ADDED', $lang_code),
                                            RESPONSE_NOTICE);
                        }
                        return true;
                    } else {
                        if ($lang_exist) {
                            $GLOBALS['app']->Session->PushLastResponse(
                                            _t('LANGUAGES_LANGUAGE_UPDATE_ERROR', $lang_code),
                                            RESPONSE_ERROR);
                        } else {
                            $GLOBALS['app']->Session->PushLastResponse(
                                            _t('LANGUAGES_LANGUAGE_ADD_ERROR', $lang_code),
                                            RESPONSE_ERROR);
                        }
                        return false;
                    }
                }
            }
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('LANGUAGES_NAME_ERROR'), RESPONSE_ERROR);
        return false;
    }

    /**
     * Returns an array of languages
     *
     * @access  public
     * @return  array   A list of available languages
     */
    function GetLanguages()
    {
        $codes = array();
        $langdir = JAWS_PATH . 'languages/';
        if (!is_dir($langdir)) {
            return $codes;
        }

        $dir = scandir($langdir);
        foreach($dir as $file) {
            if ($file != '.' && $file != '..' && !strpos($file, '.php')) {
                if (is_file($langdir.$file.'/FullName')) {
                    $fullname = implode('', @file($langdir.$file.'/FullName'));
                    if (!empty($fullname)) {
                        $codes[$file] = $fullname;
                    }
                }
            }
        }
        asort($codes);
        return $codes;
    }

    function GetGadgets()
    {
        $gadgets = array();
        $gdir = JAWS_PATH . 'gadgets/';
        if (!is_dir($gdir)) {
            return $gadgets;
        }
        $dir = scandir($gdir);
        foreach($dir as $file) {
            if ($file != '.' && $file != '..' && !strpos($file, '.php') && $file != '.svn') {
                $gadgets['gadgets/' . $file] = $file;
            }
        }
        asort($gadgets);
        return $gadgets;        
    }

    function GetPlugins()
    {
        $plugins = array();
        $pdir = JAWS_PATH . 'plugins/';
        if (!is_dir($pdir)) {
            return $plugins;
        }
        $dir = scandir($pdir);
        foreach($dir as $file) {
            if ($file != '.' && $file != '..' && !strpos($file, '.php') && $file != '.svn') {
                $plugins['plugins/' . $file] = $file;
            }
        }
        asort($plugins);
        return $plugins;        
    }

    function GetComponents()
    {
        $components = array();
        $components['Global']  = array('global'  => 'Global',
                                       'date'    => 'Date',
                                       'install' => 'Install',
                                       'upgrade' => 'Upgrade');
        $components['Gadgets'] = $this->GetGadgets();
        $components['Plugins'] = $this->GetPlugins();
        return $components;
    }

    /**
     * Returns an array of component language data
     *
     * @access  public
     * @return  array   A list of component language string
     */
    function GetLangData($component, $langTo, $langFrom)
    {
        if ($component == 'global') {
            $file = JAWS_PATH . "languages/$langTo/Global.php";
            $from_file = JAWS_PATH . "languages/$langFrom/Global.php";
        } elseif ($component == 'date') {
            $file = JAWS_PATH . "languages/$langTo/Date.php";
            $from_file = JAWS_PATH . "languages/$langFrom/Date.php";
        } elseif ($component == 'install') {
            $file = JAWS_PATH . "languages/$langTo/Install.php";
            $from_file = JAWS_PATH . "languages/$langFrom/Install.php";
        } elseif ($component == 'upgrade') {
            $file = JAWS_PATH . "languages/$langTo/Upgrade.php";
            $from_file = JAWS_PATH . "languages/$langFrom/Upgrade.php";
        } else {
            $aux = explode('/', $component);
            $file = JAWS_PATH . $component . '/languages/' . $langTo . '.php';
            $from_file = JAWS_PATH . $component . '/languages/' . $langFrom . '.php';
            if ($aux[0] == 'plugins') {
                $component = $aux[0] . '_' . $aux[1];
            } else {
                $component = $aux[1];
            }
        }

        if (!file_exists($from_file)) {
            return false;
        }

        require_once $from_file;
        $fromstrings = get_defined_constants();
        if (file_exists($file)) {
            require_once $file;
            $contents = file_get_contents($file);
        }

        $data = array();
        // Writable
        if (file_exists($file)) {
            $data['writable'] = is_writable($file);
            $data['file'] = $file;
        } else {
            $data['writable'] = is_writable(dirname($file));
            $data['file'] = dirname($file);
        }

        if ($component != 'global') {
            $global = JAWS_PATH . "languages/$langTo/Global.php";
            if (file_exists($global)) require_once $global;
        }

        if (defined('_' . strtoupper($langTo) . '_GLOBAL_LANG_DIRECTION')) {
            $data['lang_direction'] = constant('_' . strtoupper($langTo) . '_GLOBAL_LANG_DIRECTION');
        } else {
            $data['lang_direction'] = 'ltr';
        }

        // Metadata
        preg_match('/"Last-Translator:(.*)"/', isset($contents)?$contents:'', $res);
        $data['meta']['Last-Translator'] = !empty($res) ? trim($res[1]) : '';

        // Strings
        foreach ($fromstrings as $k => $v) {
            if (strpos($k, '_'.strtoupper($langFrom) . '_' . strtoupper($component)) === false) {
                continue;
            }
            $cons = str_replace('_' . strtoupper($langFrom) . '_', '', $k);
            $data['strings'][$cons][$langFrom] = $v;
            if (defined('_' . strtoupper($langTo) . '_' . $cons)) {
                $toValue = constant('_' . strtoupper($langTo) . '_' . $cons);
                if ($toValue == '') {
                    $toValue = EMPTY_STRING;
                }
                $data['strings'][$cons][$langTo] = $toValue;
            } else {
                $data['strings'][$cons][$langTo] = '';
            }
        }
        return $data;
    }

    /**
     * @access  public
     *
     * @return  boolean Success/Failure (Jaws_Error)
     */
    function SetLangData($component, $langTo, $data = null)
    {
        if ($component == 'global') {
            $file = JAWS_PATH . "languages/$langTo/Global.php";
        } elseif ($component == 'date') {
            $file = JAWS_PATH . "languages/$langTo/Date.php";
        } elseif ($component == 'install') {
            $file = JAWS_PATH . "languages/$langTo/Install.php";
        } elseif ($component == 'upgrade') {
            $file = JAWS_PATH . "languages/$langTo/Upgrade.php";
        } else {
            $aux = explode('/', $component);
            $file = JAWS_PATH . $component . '/languages/' . $langTo . '.php';
            if ($aux[0] == 'plugins') {
                $component = $aux[0] . '_' . $aux[1];
            } else {
                $component = $aux[1];
            }
        }
        $component = ucfirst($component);

        $tpl = new Jaws_Template('gadgets/Languages/templates/');
        $tpl->Load('FileTemplate.html');
        $tpl->SetBlock('template');
        $tpl->SetVariable('project', $component);
        $tpl->SetVariable('language', strtoupper($langTo));            

        // Meta
        foreach ($data['meta'] as $k => $v) {
            $tpl->SetBlock('template/meta');
            $tpl->SetVariable('key', $k);
            $tpl->SetVariable('value', str_replace('"', '\"', $v));
            $tpl->ParseBlock('template/meta');
        }

        // Strings
        foreach ($data['strings'] as $k => $v) {
            if ($v == '') {
                continue;
            } elseif ($v === EMPTY_STRING) {
                $v = '';
            }

            $tpl->SetBlock('template/string');
            $tpl->SetVariable('key', '_' . strtoupper($langTo) . '_' . $k);

            $v = preg_replace("$\r\n|\n$", '\n', $v);
            $tpl->SetVariable('value', str_replace('"', '\"', $v));
            $tpl->ParseBlock('template/string');
        }

        $tpl->ParseBlock('template');

        // Writable
        if(file_exists($file)) {
            $writeable = Jaws_Utils::is_writable($file);
        } else {
            $writeable = Jaws_Utils::is_writable(dirname($file));
        }

        if (!$writeable) {
            $GLOBALS['app']->Session->PushLastResponse(_t('LANGUAGES_NOT_PERMISSION'), RESPONSE_ERROR);
            return false;
        }

        if (Jaws_Utils::file_put_contents($file, $tpl->Get())) {
            $GLOBALS['app']->Session->PushLastResponse(_t('LANGUAGES_UPDATED', $component), RESPONSE_NOTICE);
            return true;
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('LANGUAGES_NOT_UPDATED', $component), RESPONSE_ERROR);
            return false;
        }
    }
}
?>
