<?php
/**
 * LinkDump Gadget Admin
 *
 * @category   GadgetModel
 * @package    LinkDump
 * @author     Amir Mohammad Saied <amirsaied@gmail.com>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2005-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/gpl.html
 */
require_once JAWS_PATH . 'gadgets/LinkDump/Model.php';
class LinkDumpAdminModel extends LinkDumpModel
{
    /**
     * Installs the gadget
     *
     * @access  public
     * @return  boolean Returns true on a successfull Install and Jaws_Error on errors
     */
    function InstallGadget()
    {
        $new_dir = JAWS_DATA . 'xml' . DIRECTORY_SEPARATOR;
        if (!Jaws_Utils::mkdir($new_dir)) {
            return new Jaws_Error(_t('GLOBAL_ERROR_FAILED_CREATING_DIR', $new_dir), _t('LINKDUMP_NAME'));
        }

        $result = $this->installSchema('schema.xml');
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        $result = $this->installSchema('insert.xml', '', 'schema.xml', true);
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        // Registry key
        $GLOBALS['app']->Registry->NewKey('/gadgets/LinkDump/max_limit_count', '100');
        $GLOBALS['app']->Registry->NewKey('/gadgets/LinkDump/links_target', 'blank');

        return true;
    }

    /**
     * Uninstalls the gadget
     *
     * @access  public
     * @return  boolean  Success/Failure (Jaws_Error)
     */
    function UninstallGadget()
    {
        $tables = array('linkdump_links',
                        'linkdump_groups',
                        'linkdump_tags',
                        'linkdump_links_tags');
        foreach ($tables as $table) {
            $result = $GLOBALS['db']->dropTable($table);
            if (Jaws_Error::IsError($result)) {
                $gName  = _t('LINKDUMP_NAME');
                $errMsg = _t('GLOBAL_ERROR_GADGET_NOT_UNINSTALLED', $gName);
                $GLOBALS['app']->Session->PushLastResponse($errMsg, RESPONSE_ERROR);
                return new Jaws_Error($errMsg, $gName);
            }
        }

        // registry keys
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/LinkDump/max_limit_count');
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/LinkDump/links_target');

        return true;
    }

    /**
     * Update the gadget
     *
     * @access  public
     * @param   string  $old    Current version (in registry)
     * @param   string  $new    New version (in the $gadgetInfo file)
     * @return  boolean  Success/Failure (Jaws_Error)
     */
    function UpdateGadget($old, $new)
    {
        $result = $this->installSchema('schema.xml', '', "$old.xml");
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        if (version_compare($old, '0.4.0', '<')) {
            $result = $this->installSchema('insert.xml', '', 'schema.xml', true);
            if (Jaws_Error::IsError($result)) {
                return $result;
            }

            // ACL keys
            $GLOBALS['app']->ACL->NewKey('/ACL/gadgets/LinkDump/ManageLinks', 'true');
            $GLOBALS['app']->ACL->NewKey('/ACL/gadgets/LinkDump/ManageGroups', 'true');
            $GLOBALS['app']->ACL->NewKey('/ACL/gadgets/LinkDump/ManageTags',   'true');
            $GLOBALS['app']->ACL->DeleteKey('/ACL/gadgets/LinkDump/AddLink');
            $GLOBALS['app']->ACL->DeleteKey('/ACL/gadgets/LinkDump/UpdateLink');
            $GLOBALS['app']->ACL->DeleteKey('/ACL/gadgets/LinkDump/DeleteLink');

            // Registry keys.
            $GLOBALS['app']->Registry->NewKey('/gadgets/LinkDump/max_limit_count', '100');
            $GLOBALS['app']->Registry->NewKey('/gadgets/LinkDump/links_target', 'blank');
            $GLOBALS['app']->Registry->DeleteKey('/gadgets/LinkDump/limitation');
            $GLOBALS['app']->Registry->DeleteKey('/gadgets/LinkDump/target');
        }

        return true;
    }

    /**
     * Update link's information
     *
     * @access  public
     * @param   int     $id             The id of link
     * @param   string  $title          Title of the link
     * @param   string  $description    Link's description
     * @param   string  $url            Link's URL
     * @return  boolean True on success and Jaws_Error in otherwise
     */
    function UpdateLink($id, $gid, $title, $url, $desc, $tags, $rank)
    {
        $oldLink = $this->GetLink($id);
        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        MDB2::loadFile('Date');
        $params                = array();
        $params['id']          = (int)$id;
        $params['gid']         = (int)$gid;
        $params['title']       = $xss->parse($title);
        $params['description'] = $xss->parse($desc);
        $params['url']         = $url;
        $params['now']         = MDB2_Date::mdbNow();
        $params['rank']        = $rank;

        $sql = '
            UPDATE [[linkdump_links]] SET
                [title]       = {title},
                [gid]         = {gid},
                [description] = {description},
                [url]         = {url},
                [updatetime]  = {now},
                [rank]        = {rank}
            WHERE [id] = {id}';

        $res = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_UPDATE_ERROR'), RESPONSE_ERROR);
            return new Jaws_Error($res->getMessage(), 'SQL');
        }

        //$this->MoveLink($id, $gid, $oldLink['gid'], $rank, $oldLink['rank']);

        $sql = 'SELECT [tag_id] FROM [[linkdump_links_tags]] WHERE [link_id] = {id}';
        $res = $this->GetLink((int)$id);

        $prev_tags = array();
        $tmp = split(",", $res['tags']);
        foreach ($tmp as $ptmp) {
            $prev_tags[] = trim($ptmp);
        }

        $current_tags = array();
        $tags = $xss->parse($tags);
        $tags = split(",", $tags);
        if (is_array($tags) && count($tags) > 0) {
            foreach ($tags as $tag) {
                $current_tags[] = trim($tag);
            }
        }

        $to_be_added_tags = array_diff($current_tags, $prev_tags);
        foreach ($to_be_added_tags as $newtag) {
            $this->AddTagToLink((int)$id, $newtag);
        }

        $to_be_removed_tags = array_diff($prev_tags, $current_tags);
        foreach ($to_be_removed_tags as $oldtag) {
            $this->RemoveTagFromLink((int)$id, $oldtag);
        }

        if ($oldLink['gid'] != $gid) {
            $this->PopulateFeed($oldLink['gid']);
        }
        $this->PopulateFeed($gid);

        $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_UPDATED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * function for change gid, pid and rank of menus
     *
     * @access  public
     * @return  array   Response (notice or error)
     */
    function MoveLink($lid, $new_gid, $old_gid, $new_rank, $old_rank)
    {

        if ($new_gid != $old_gid) {
            // resort menu items in old_pid
            $sql = '
                UPDATE [[linkdump_links]] SET
                    [rank] = [rank] - 1
                WHERE
                    [gid] = {gid}
                  AND
                    [rank] > {rank}';

            $params         = array();
            $params['gid']  = $old_gid;
            $params['rank'] = $old_rank;
            $res = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($res)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
                return false;
            }
        }

        if ($new_gid != $old_gid) {
            // resort menu items in new_pid
            $sql = '
                UPDATE [[linkdump_links]] SET
                    [rank] = [rank] + 1
                WHERE
                    [id] <> {lid}
                  AND
                    [gid] = {gid}
                  AND
                    [rank] >= {new_rank}';

            $params             = array();
            $params['lid']      = $lid;
            $params['gid']      = $new_gid;
            $params['new_rank'] = $new_rank;
            $res = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($res)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
                return false;
            }
        } elseif (empty($old_rank)) {
            $sql = '
                UPDATE [[linkdump_links]] SET
                    [rank] = [rank] + 1
                WHERE
                    [id] <> {lid}
                  AND
                    [gid] = {gid}
                  AND
                    [rank] >= {new_rank}';

            $params             = array();
            $params['lid']      = $lid;
            $params['gid']      = $new_gid;
            $params['new_rank'] = $new_rank;
            $res = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($res)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
                return false;
            }
        } elseif ($new_rank > $old_rank) {
            // resort menu items in new_pid
            $sql = '
                UPDATE [[linkdump_links]] SET
                    [rank] = [rank] - 1
                WHERE
                    [id] <> {lid}
                  AND
                    [gid] = {gid}
                  AND
                    [rank] > {old_rank}
                  AND
                    [rank] <= {new_rank}';

            $params             = array();
            $params['lid']      = $lid;
            $params['gid']      = $new_gid;
            $params['old_rank'] = $old_rank;
            $params['new_rank'] = $new_rank;
            $res = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($res)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
                return false;
            }
        } elseif ($new_rank < $old_rank) {
            // resort menu items in new_pid
            $sql = '
                UPDATE [[linkdump_links]] SET
                    [rank] = [rank] + 1
                WHERE
                    [id] <> {lid}
                  AND
                    [gid] = {gid}
                  AND
                    [rank] >= {new_rank}
                  AND
                    [rank] < {old_rank}';

            $params             = array();
            $params['lid']      = $lid;
            $params['gid']      = $new_gid;
            $params['old_rank'] = $old_rank;
            $params['new_rank'] = $new_rank;
            $res = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($res)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
                return false;
            }
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_REPLACED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * This is the short Description for the Function
     *
     * This is the long description for the Class
     *
     * @return	mixed	 Description
     * @access	public
     */
    function AddTagToLink($link, $tag)
    {
        $tag = str_replace(" ", "_", strtolower(trim($tag)));
        $sql = 'SELECT [id] FROM [[linkdump_tags]] WHERE [tag] = {tag}';
        $res = $GLOBALS['db']->queryRow($sql, array('tag' => $tag));
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_ADD_TAG_ERROR'), RESPONSE_ERROR);
            return new Jaws_Error($rs1->getMessage(), 'SQL');
        } elseif (empty($res['id'])) {
            $sql = 'INSERT INTO [[linkdump_tags]] ([tag]) VALUES({tag})';
            $rs1 = $GLOBALS['db']->query($sql, array('tag' => $tag));
            if (Jaws_Error::IsError($rs1)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_ADD_TAG_ERROR'), RESPONSE_ERROR);
                return new Jaws_Error($rs1->getMessage(), 'SQL');
            }
            $sql    = 'SELECT MAX([id]) FROM [[linkdump_tags]] WHERE [tag] = {tag}';
            $tag_id = $GLOBALS['db']->queryOne($sql, array('tag' => $tag));
            if (Jaws_Error::IsError($tag_id)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_ADD_TAG_ERROR'), RESPONSE_ERROR);
                return new Jaws_Error($rs1->getMessage(), 'SQL');
            }
        } else {
            $tag_id = $res['id'];
        }

        $params             = array();
        $params['tag_id']   = (int)$tag_id;
        $params['link_id']  = (int)$link;

        $sql = '
            SELECT
                COUNT([tag_id])
            FROM [[linkdump_links_tags]]
            WHERE [tag_id] = {tag_id} AND [link_id] = {link_id}';
        $rs = $GLOBALS['db']->queryOne($sql, $params);
        if (Jaws_Error::IsError($rs)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_ADD_TAG_ERROR'), RESPONSE_ERROR);
            return new Jaws_Error($rs->getMessage(), 'SQL');
        } elseif ((int)$rs === 0) {
            $sql = 'INSERT INTO [[linkdump_links_tags]] ([tag_id], [link_id]) VALUES({tag_id}, {link_id})';
            $rs2 = $GLOBALS['db']->query($sql, $params);
        }

        return true;
    }

    /**
    * This is the short Description for the Function
    *
    * @return mixed	 Description
    * @access public
    */
    function RemoveTagFromLink($id, $tag)
    {
        $tag = str_replace(' ', '_', strtolower(trim($tag)));
        $sql = 'SELECT [id] FROM [[linkdump_tags]] WHERE [tag] = {tag}';
        $res = $GLOBALS['db']->queryRow($sql, array('tag' => trim($tag)));
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_UPDATE_ERROR'), RESPONSE_ERROR);
            return new Jaws_Error($res->getMessage(), 'SQL');
        }
        if (isset($res['id'])) {
            $params             = array();
            $params['link_id']  = (int)$id;
            $params['tag_id']   = $res['id'];

            $sql = 'DELETE FROM [[linkdump_links_tags]] WHERE [link_id] = {link_id} AND [tag_id] = {tag_id}';
            $res = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($res)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_UPDATE_ERROR'), RESPONSE_ERROR);
                return new Jaws_Error($res->getMessage(), 'SQL');
            }
        }

        return true;
    }

    /**
     * Delete link
     *
     * @access  public
     * @param   int $id Link's id
     * @return  Boolean True on success on Jaws_Error otherwise
     */
    function DeleteLink($id, $gid = '')
    {
        $params       = array();
        $params['id'] = $id;

        $sql = 'DELETE FROM [[linkdump_links]] WHERE [id] = {id}';
        $res  = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_DELETE_ERROR'), RESPONSE_ERROR);
            return new Jaws_Error($res->getMessage(), 'SQL');
        }

        $sql = 'DELETE FROM [[linkdump_links_tags]] WHERE [link_id] = {id}';
        $res = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_DELETE_ERROR'), RESPONSE_ERROR);
            return new Jaws_Error($res->getMessage(), 'SQL');
        }

        $this->PopulateFeed($gid);
        $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_DELETED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Will Poupulate the linkdump feed
     *
     * @access  public
     */
    function PopulateFeed($gid)
    {
        ///FIXME maybe it would be good to put some error checking here and return some messages if there's an error
        $group = $this->GetGroup($gid);
        if (Jaws_Error::IsError($group) || empty($group) || !isset($group['id'])) {
            return false;
        }

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $feedname = empty($group['fast_url']) ?
                    $GLOBALS['app']->UTF8->str_replace(' ', '-', $group['title']) : $xss->filter($group['fast_url']);
        $feedname = preg_replace('/[@?^=%&:;\/~\+# ]/i', '\1', $feedname);

        $html = $GLOBALS['app']->LoadGadget('LinkDump', 'HTML');
        @file_put_contents(JAWS_DATA. "xml/linkdump.$feedname.rdf", $html->PopulateFeed($gid));
        Jaws_Utils::chmod(JAWS_DATA. "xml/linkdump.$feedname.rdf");

        return true;
    }

    /**
    * Insert a group
    * @access  public
    *
    * @return  boolean Success/Failure (Jaws_Error)
    */
    function InsertGroup($title, $fast_url, $limit_count, $link_type)
    {
        $fast_url = empty($fast_url) ? $title : $fast_url;
        $fast_url = $this->GetRealFastUrl($fast_url, 'linkdump_groups');

        $sql = '
            INSERT INTO [[linkdump_groups]]
                ([title], [fast_url], [limit_count], [link_type])
            VALUES
                ({title}, {fast_url}, {limit_count}, {link_type})';

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $params = array();
        $params['title']       = $xss->parse($title);
        $params['fast_url']    = $xss->parse($fast_url);
        $params['limit_count'] = $limit_count;
        $params['link_type']   = $link_type;
        $res = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }
        $gid = $GLOBALS['db']->lastInsertID('linkdump_groups', 'id');
        $GLOBALS['app']->Session->PushLastResponse($gid.'%%' . _t('LINKDUMP_GROUPS_ADDED'), RESPONSE_NOTICE);

        return true;
    }

    /**
    * Insert a link
    * @access  public
    *
    * @return  boolean Success/Failure (Jaws_Error)
    */
    function InsertLink($gid, $title, $url, $desc, $tags, $rank)
    {
        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        MDB2::loadFile('Date');
        $params                     = array();
        $params['title']            = $xss->parse($title);
        $params['description']      = $xss->parse($desc);
        $params['url']              = $url;
        $params['gid']              = $gid;
        $params['rank']             = $rank;
        $params['now']              = MDB2_Date::mdbNow();

        $sql = '
            INSERT INTO [[linkdump_links]]
                ([gid], [title], [description], [url], [createtime], [updatetime], [rank])
            VALUES
                ({gid}, {title}, {description}, {url}, {now}, {now}, {rank})';

        $rs = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($rs)) {
            $GLOBALS['app']->Session->PushLastResponse($rs->GetMessage(), RESPONSE_ERROR);
            return new Jaws_Error(_t('LINKDUMP_LINKS_ADD_ERROR', 'AddLink'), _t('LINKDUMP_NAME'));
        }

        $lid = $GLOBALS['db']->lastInsertID('linkdump_links', 'id');
        //$this->MoveLink($lid, $gid, $gid, $rank, null);

        $sql = 'SELECT MAX([id]) FROM [[linkdump_links]] WHERE [url] = {url}';
        $max = $GLOBALS['db']->queryOne($sql, array('url' => $url));
        if (Jaws_Error::IsError($max)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_ADD_ERROR'), RESPONSE_ERROR);
            return new Jaws_Error(_t('LINKDUMP_ERROR_LINK_NOT_ADDED', 'AddLink'), _t('LINKDUMP_NAME'));
        }

        if ($max) {
            if (!empty($tags)) {
                $tags = $xss->parse($tags);
                $tags = split(",", $tags);
                if (is_array($tags) && count($tags) > 0) {
                    foreach ($tags as $tag) {
                        $res = $this->AddTagToLink($max, $tag);
                        if (Jaws_Error::IsError($res)) {
                            $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_LINKS_ADD_TAG_ERROR'), RESPONSE_ERROR);
                            return $res;
                        }
                    }
                }
            }
        }

        $this->PopulateFeed($gid);
        $GLOBALS['app']->Session->PushLastResponse($lid .'%%' . _t('LINKDUMP_LINKS_ADDED'), RESPONSE_NOTICE);
        return true;
    }

    /**
    * Update a group
    * @access  public
    *
    * @return  boolean Success/Failure (Jaws_Error)
    */
    function UpdateGroup($gid, $title, $fast_url, $limit_count, $link_type)
    {
        $fast_url = empty($fast_url) ? $title : $fast_url;
        $fast_url = $this->GetRealFastUrl($fast_url, 'linkdump_groups', false);

        $sql = '
            UPDATE [[linkdump_groups]] SET
                [title]       = {title},
                [fast_url]    = {fast_url},
                [limit_count] = {limit_count},
                [link_type]   = {link_type}
            WHERE [id] = {gid}';

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $params                = array();
        $params['gid']         = $gid;
        $params['title']       = $xss->parse($title);
        $params['fast_url']    = $xss->parse($fast_url);
        $params['limit_count'] = $limit_count;
        $params['link_type']   = $link_type;
        $res = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }
        $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_GROUPS_UPDATED'), RESPONSE_NOTICE);

        return true;
    }

    /**
     * Delete a group
     *
     * @access  public
     * @return  boolean True if query was successful and Jaws_Error on error
     */
    function DeleteGroup($gid)
    {
        if ($gid == 1) {
            $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_GROUPS_NOT_DELETABLE'), RESPONSE_ERROR);
            return false;
        }
        $group = $this->GetGroup($gid);
        if (Jaws_Error::IsError($group)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        if(!isset($group['id'])) {
            $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_GROUPS_NOT_EXISTS'), RESPONSE_ERROR);
            return false;
        }

        $links = $this->GetGroupLinks($gid);
        foreach ($links as $link) {
            $sql = 'DELETE FROM [[linkdump_links_tags]] WHERE [link_id] = {id}';
            $res = $GLOBALS['db']->query($sql, array('id' => $link['id']));
            if (Jaws_Error::IsError($res)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
                return false;
            }
        }

        $sql = 'DELETE FROM [[linkdump_links]] WHERE [gid] = {gid}';
        $res = $GLOBALS['db']->query($sql, array('gid' => $gid));
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        $sql = 'DELETE FROM [[linkdump_groups]] WHERE [id] = {gid}';
        $res = $GLOBALS['db']->query($sql, array('gid' => $gid));
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('GLOBAL_ERROR_QUERY_FAILED'), RESPONSE_ERROR);
            return false;
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('LINKDUMP_GROUPS_DELETED', $gid), RESPONSE_NOTICE);

        return true;
    }

    /**
     * Implements the GetRealFastURL
     *
     * @access  protected
     * @param   string     $fast_url     Fast URL
     * @param   string     $table        DB table name (with no [[ ]])
     * @param   boolean    $unique_check must be false in update methods
     * @param   string     $field        Table field where fast_url is stored
     * @return  string     Correct fast URL
     */
    function GetRealFastURL($fast_url, $table, $unique_check = true, $field = 'fast_url')
    {
        if (is_numeric($fast_url)) {
            $fast_url = '-' . $fast_url . '-';
        }
        $fast_url = $GLOBALS['app']->UTF8->str_replace(' ',  '-', $fast_url);
        $fast_url = $GLOBALS['app']->UTF8->str_replace('/',  '-', $fast_url);
        $fast_url = $GLOBALS['app']->UTF8->str_replace('\\', '-', $fast_url);

        if (!$unique_check) {
            return $fast_url;
        }

        $params = array();
        $params['fast_url'] = $fast_url;

        $sql = "
             SELECT COUNT(*)
             FROM [[$table]]
             WHERE [$field] = {fast_url}";

        $total = $GLOBALS['db']->queryOne($sql, $params);
        if (Jaws_Error::isError($total) || ($total == '0')) {
            return $fast_url;
        }

        //Get a list of fast_url's that match the current entry
        $params['fast_url'] = $GLOBALS['app']->UTF8->trim($fast_url).'%';

        $sql = "
             SELECT [$field]
             FROM [[$table]]
             WHERE [$field] LIKE {fast_url}";

        $urlList = $GLOBALS['db']->queryAll($sql, $params);
        if (Jaws_Error::IsError($urlList)) {
            return $fast_url;
        }

        $counter = 1;
        $numbers = array();
        foreach($urlList as $url) {
            //Matches the foo-\d?
            if (preg_match("/(.+?)-([0-9]{0,})/", $url[$field], $matches)) {
                $numbers[] = (int)$matches[2];
            }
        }
        if (count($numbers) == 0) {
            return $fast_url . '-1';
        }
        $lastNumber = $numbers[count($numbers)-1];
        return $fast_url.'-'.($lastNumber+1);
    }
}