<?php
/**
 * Settings Core Gadget
 *
 * @category   GadgetModel
 * @package    Settings
 * @author     Jonathan Hernandez <ion@suavizado.com>
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2004-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
class SettingsAdminModel extends Jaws_Model
{
    var $_Name = 'Settings';

    /**
     * Installs the gadget
     *
     * @access       public
     * @return       true on successful installation, Jaws_Error otherwise
     */
    function InstallGadget()
    {
        $GLOBALS['app']->Registry->NewKey('/gadgets/Settings/pluggable', 'false');
        return true;
    }

    /**
     * Returns an array of languages
     *
     * @access  public
     * @return  array   A list of available languages
     */
    function GetLanguageList()
    {
        $codes = array();
        $langdir = JAWS_PATH . 'languages/';
        if (!is_dir($langdir)) {
            return $codes;
        }

        $dir = scandir($langdir);
        foreach($dir as $file) {
            if ($file != '.' && $file != '..' && !strpos($file, '.php')) {
                if (is_file($langdir.$file.'/FullName')) {
                    $fullname = implode('', @file($langdir.$file.'/FullName'));
                    if (!empty($fullname)) {
                        $codes[$file] = $fullname;
                    }
                }
            }
        }
        asort($codes);
        return $codes;
    }

    /**
     * Get the available calendars
     *
     * @access   public
     * @return   Array   Array with available calendars and Jaws_Error otherwise
     */
    function GetCalendarList()
    {
        $calendars = array();
        $path = JAWS_PATH . 'include/Jaws/Date';
        if (is_dir($path)) {
            $dir = scandir($path);
            foreach ($dir as $calendar) {
                if (stristr($calendar, '.php')) {
                    $calendar = str_replace('.php', '', $calendar);
                    $calendars[$calendar] = $calendar;
                }
            }

            return $calendars;
        }

        return false;
    }

    /**
     * Get the available editors
     *
     * @access   public
     * @return   Array   Array with available editors and Jaws_Error otherwise
     */
    function GetEditorList()
    {
        $editors = array();
        $editors['TextArea'] = _t('SETTINGS_EDITOR_CLASSIC');
        $editors['TinyMCE']  = _t('SETTINGS_EDITOR_FRIENDLY');

        return $editors;
    }

    /**
     * Get the available date formats
     *
     * @access   public
     * @return   Array   Array with available date formats and Jaws_Error otherwise
     */
    function GetDateFormatList()
    {
        $dt_formats = array();
        $time = time();
        $date = $GLOBALS['app']->loadDate();
        $dt_formats['MN j, g:ia']     = $date->Format($time, 'MN j, g:ia');
        $dt_formats['j.m.y']          = $date->Format($time, 'j.m.y');
        $dt_formats['j MN, g:ia']     = $date->Format($time, 'j MN, g:ia');
        $dt_formats['y.m.d, g:ia']    = $date->Format($time, 'y.m.d, g:ia');
        $dt_formats['d MN Y']         = $date->Format($time, 'd MN Y');
        $dt_formats['DN d MN Y']      = $date->Format($time, 'DN d MN Y');
        $dt_formats['DN d MN Y g:ia'] = $date->Format($time, 'DN d MN Y g:ia');
        $dt_formats['j MN y']         = $date->Format($time, 'j MN y');
        $dt_formats['j m Y - H:i']    = $date->Format($time, 'j m Y - H:i');
        $dt_formats['AGO']            = $date->Format($time, 'since');

        return $dt_formats;
    }

    /**
     * Get the timezones list
     *
     * @access   public
     * @return   Array   Array with timezone and Jaws_Error otherwise
     */
    function GetTimeZonesList()
    {
        $timezones          = array();
        $timezones['-12']   = '[UTC - 12] Baker Island, Howland Island';
        $timezones['-11']   = '[UTC - 11] Midway Island, Samoa';
        $timezones['-10']   = '[UTC - 10] Hawaii';
        $timezones['-9.5']  = '[UTC - 9:30] Marquesa Islands, Taiohae';
        $timezones['-9']    = '[UTC - 9] Alaska';
        $timezones['-8']    = '[UTC - 8] Pacific Time (US &amp; Canada), Tijuana';
        $timezones['-7']    = '[UTC - 7] Mountain Time (US &amp; Canada), Arizona';
        $timezones['-6']    = '[UTC - 6] Central Time (US &amp; Canada), Mexico City';
        $timezones['-5']    = '[UTC - 5] Eastern Time (US &amp; Canada), Bogota, Lima, Quito';
        $timezones['-4']    = '[UTC - 4] Atlantic Time (Canada), Caracas, La Paz, Santiago';
        $timezones['-3.5']  = '[UTC - 3:30] Newfoundland';
        $timezones['-3']    = '[UTC - 3] Brasilia, Buenos Aires, Georgetown, Greenland';
        $timezones['-2']    = '[UTC - 2] Mid-Atlantic, Ascension Islands, St. Helena';
        $timezones['-1']    = '[UTC - 1] Azores, Cape Verde Islands';
        $timezones['0']     = '[UTC] Western European, Casablanca, Lisbon, London';
        $timezones['1']     = '[UTC + 1] Amsterdam, Berlin, Brussels, Madrid, Paris, Rome';
        $timezones['2']     = '[UTC + 2] Cairo, Helsinki, Kaliningrad, South Africa';
        $timezones['3']     = '[UTC + 3] Baghdad, Riyadh, Moscow, St. Petersburg, Nairobi';
        $timezones['3.5']   = '[UTC + 3:30] Tehran';
        $timezones['4']     = '[UTC + 4] Abu Dhabi, Baku, Muscat, Tbilisi';
        $timezones['4.5']   = '[UTC + 4:30] Kabul';
        $timezones['5']     = '[UTC + 5] Ekaterinburg, Islamabad, Karachi, Tashkent';
        $timezones['5.5']   = '[UTC + 5:30] Bombay, Calcutta, Madras, New Delhi';
        $timezones['5.75']  = '[UTC + 5:45] Kathmandu';
        $timezones['6']     = '[UTC + 6] Almaty, Colombo, Dhaka, Novosibirsk';
        $timezones['6.5']   = '[UTC + 6:30] Rangoon, Cocos Islands';
        $timezones['7']     = '[UTC + 7] Bangkok, Hanoi, Jakarta, Krasnoyarsk';
        $timezones['8']     = '[UTC + 8] Beijing, Hong Kong, Perth, Singapore, Taipei';
        $timezones['8.75']  = '[UTC + 8:45] Western Australia';
        $timezones['9']     = '[UTC + 9] Osaka, Sapporo, Seoul, Tokyo, Yakutsk';
        $timezones['9.5']   = '[UTC + 9:30] Adelaide, Darwin, Yakutsk';
        $timezones['10']    = '[UTC + 10] Canberra, Guam, Melbourne, Sydney, Vladivostok';
        $timezones['10.5']  = '[UTC + 10:30] Lord Howe Island, South Australia';
        $timezones['11']    = '[UTC + 11] Magadan, New Caledonia, Solomon Islands';
        $timezones['11.5']  = '[UTC + 11:30] Norfolk Island';
        $timezones['12']    = '[UTC + 12] Auckland, Fiji, Kamchatka, Marshall Islands';
        $timezones['12.75'] = '[UTC + 12:45] Chatham Islands';
        $timezones['13']    = '[UTC + 13] Tonga, Phoenix Islands';
        $timezones['14']    = '[UTC + 14] Kiribati';
        return $timezones;
    }

    /**
     * Save the basic settings
     *
     * @access  public
     * @param   array   $settings  Settings array. This should have the following entries:
     *
     * $settings = array(
     *                   'site_name',        => //Site name
     *                   'site_description', => //Site description
     *                   'url',              => //Site url
     *                   'site_language',    => //Default site language
     *                   'admin_language',   => //Admin area language
     *                   'main_gadget',      => //Main gadget
     *                   'owner_name',       => //Owner name
     *                   'owner_email',      => //Owner email,
     *                   'copyright',        => //Copyright,
     *                  );
     *
     * @return  boolean True or Jaws_Error
     */
    function SaveBasicSettings($settings)
    {
        $basicKeys = array('site_status', 'site_name', 'site_description', 'site_keywords', 'site_language',
                           'admin_language', 'main_gadget', 'owner_name', 'owner_email',
                           'copyright');

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $settings['owner_name']
            = isset($settings['owner_name']) ?
              $xss->parse($settings['owner_name']) : '';
        $settings['owner_email']
            = isset($settings['owner_email']) ?
              $xss->parse($settings['owner_email']) : '';
        $settings['copyright']
            = isset($settings['copyright']) ?
              $xss->parse($settings['copyright']) : '';

        foreach ($settings as $settingKey => $settingValue) {
            if (!in_array($settingKey, $basicKeys)) {
                continue;
            }
            $GLOBALS['app']->Registry->Set('/config/' . $settingKey, $settingValue);
        }
        $GLOBALS['app']->Registry->Commit('core');
        $GLOBALS['app']->Session->PushLastResponse(_t('SETTINGS_SAVED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Save the advanced settings
     *
     * @access  public
     * @param   array   $settings  Settings array. This should have the following entries:
     *
     * $settings = array(
     *                   'date_format',     //Date format
     *                   'calendar_type',   //Date Calendar
     *                   'use_gravatar',    //Use gravatar service?
     *                   'gravatar_rating', //Gravatar rating
     *                   'allow_comments',  //Allow comments?
     *                   'show_viewsite',   //show the view site on CP?
     *                   'title_separator', //Separator used when user uses page_title
     *                   'editor',          //Editor to use
     *                  );
     *
     * @return  boolean True or Jaws_Error
     */
    function SaveAdvancedSettings($settings)
    {
        $advancedKeys = array('date_format', 'calendar_type', 'calendar_language',
                              'use_gravatar', 'gravatar_rating', 'allow_comments',
                              'show_viewsite', 'title_separator', 'editor', 'timezone');

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');

        foreach ($settings as $settingKey => $settingValue) {
            if (!in_array($settingKey, $advancedKeys)) {
                continue;
            }

            if (is_string($settingValue) && !empty($settingValue)) {
                $settingValue = $xss->parse($settingValue);
            }
            $GLOBALS['app']->Registry->Set('/config/' . $settingKey, $settingValue);
        }
        $GLOBALS['app']->Registry->Commit('core');
        $GLOBALS['app']->Session->PushLastResponse(_t('SETTINGS_SAVED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Save the mail settings
     *
     * @access  public
     * @param   array   $settings  Settings array. This should have the following entries:
     *
     * $settings = array(
     *                   'mailer',
     *                   'from_email',
     *                   'from_name',
     *                   'sendmail_path',
     *                   'smtp_host',
     *                   'smtp_port',
     *                   'smtp_auth',
     *                   'smtp_user',
     *                   'smtp_pass',
     *                  );
     *
     * @return  boolean True or Jaws_Error
     */
    function UpdateMailSettings($settings)
    {
        $mailKeys = array('mailer', 'from_email', 'from_name', 'sendmail_path', 
                          'smtp_host', 'smtp_port', 'smtp_auth', 'smtp_user', 'smtp_pass');

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        foreach ($settings as $settingKey => $settingValue) {
            if (!in_array($settingKey, $mailKeys)) {
                continue;
            }

            if (is_string($settingValue) && !empty($settingValue)) {
                $settingValue = $xss->parse($settingValue);
            }
            if ($settingKey == 'smtp_pass' && empty($settingValue)) {
                continue;
            }

            $GLOBALS['app']->Registry->Set('/network/' . $settingKey, $settingValue);
        }
        $GLOBALS['app']->Registry->Commit('core');
        $GLOBALS['app']->Session->PushLastResponse(_t('SETTINGS_SAVED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Save the ftp settings
     *
     * @access  public
     * @param   array   $settings  Settings array. This should have the following entries:
     *
     * $settings = array(
     *                   'ftp_enabled',
     *                   'ftp_host',
     *                   'ftp_port',
     *                   'ftp_mode',
     *                   'ftp_user',
     *                   'ftp_pass',
     *                   'ftp_root',
     *                  );
     *
     * @return  boolean True or Jaws_Error
     */
    function UpdateFTPSettings($settings)
    {
        $ftpKeys = array('ftp_enabled', 'ftp_host', 'ftp_port',
                         'ftp_mode', 'ftp_user', 'ftp_pass', 'ftp_root');

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        foreach ($settings as $settingKey => $settingValue) {
            if (!in_array($settingKey, $ftpKeys)) {
                continue;
            }

            if (is_string($settingValue) && !empty($settingValue)) {
                $settingValue = $xss->parse($settingValue);
            }
            if ($settingKey == 'ftp_pass' && empty($settingValue)) {
                continue;
            }

            $GLOBALS['app']->Registry->Set('/network/' . $settingKey, $settingValue);
        }
        $GLOBALS['app']->Registry->Commit('core');
        $GLOBALS['app']->Session->PushLastResponse(_t('SETTINGS_SAVED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Save the proxy settings
     *
     * @access  public
     * @param   array   $settings  Settings array. This should have the following entries:
     *
     * $settings = array(
     *                   'proxy_enabled',
     *                   'proxy_host',
     *                   'proxy_port',
     *                   'proxy_auth',
     *                   'proxy_user',
     *                   'proxy_pass',
     *                  );
     *
     * @return  boolean True or Jaws_Error
     */
    function UpdateProxySettings($settings)
    {
        $proxyKeys = array('proxy_enabled', 'proxy_host', 'proxy_port',
                         'proxy_auth', 'proxy_user', 'proxy_pass');

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        foreach ($settings as $settingKey => $settingValue) {
            if (!in_array($settingKey, $proxyKeys)) {
                continue;
            }

            if (is_string($settingValue) && !empty($settingValue)) {
                $settingValue = $xss->parse($settingValue);
            }
            if ($settingKey == 'proxy_pass' && empty($settingValue)) {
                continue;
            }

            $GLOBALS['app']->Registry->Set('/network/' . $settingKey, $settingValue);
        }
        $GLOBALS['app']->Registry->Commit('core');
        $GLOBALS['app']->Session->PushLastResponse(_t('SETTINGS_SAVED'), RESPONSE_NOTICE);
        return true;
    }
}
