<?php
/**
 * StaticPage Gadget Admin
 *
 * @category   GadgetAdmin
 * @package    StaticPage
 * @author     Jon Wood      <jon@jellybob.co.uk>
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @copyright  2004-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/gpl.html
 */
class StaticPageAdminHTML extends Jaws_GadgetHTML
{
    /**
     * Constructor
     *
     * @access public
     */
    function StaticPageAdminHTML()
    {
        $this->Init('StaticPage');
    }

    /**
     * Builds the menubar
     *
     * @access       public
     * @param        string  $selected Selected action
     * @return       string  The html menubar
     */
    function MenuBar($selected)
    {
        $actions = array('Admin', 'AddNewPage','PreviewAddPage', 'Properties');
        if (!in_array($selected, $actions)) {
            $selected = 'Admin';
        }

        if ($selected == 'PreviewAddPage') {
            $selected = 'AddNewPage';
        }

        require_once JAWS_PATH . 'include/Jaws/Widgets/Menubar.php';
        $menubar = new Jaws_Widgets_Menubar();
        if ($this->GetPermission('default')) {
            $menubar->AddOption('Admin', _t('STATICPAGE_MENU_PAGES'),
                                BASE_SCRIPT . '?gadget=StaticPage&amp;action=Admin', STOCK_DOCUMENTS);
        }

        if ($this->GetPermission('AddPage')) {
            $menubar->AddOption('AddNewPage', _t('STATICPAGE_MENU_ADDPAGE'),
                                BASE_SCRIPT . '?gadget=StaticPage&amp;action=AddNewPage', STOCK_NEW);
        }
        
        if ($this->GetPermission('Properties')) {
            $menubar->AddOption('Properties', _t('GLOBAL_SETTINGS'),
                                BASE_SCRIPT . '?gadget=StaticPage&amp;action=Properties', 
                                'images/stock/properties.png');
        }
        $menubar->Activate($selected);

        return $menubar->Get();
    }

    /**
     * Builds the 'manage settings/properties' view
     *
     * @access  public
     * @return  string  XHTML of settings view
     */
    function Properties()
    {
        $this->CheckPermission('default');
        $this->AjaxMe('script.js');

        $tpl = new Jaws_Template('gadgets/StaticPage/templates/');
        $tpl->Load('AdminStaticPage.html');
        $tpl->SetBlock('Properties');

        $request =& Jaws_Request::getInstance();
        $action  = $request->getRaw('action', 'get');
        $tpl->SetVariable('menubar', $this->MenuBar($action));

        $model = $GLOBALS['app']->loadGadget('StaticPage', 'Model');
        
        //Build the form
        $form =& Piwi::CreateWidget('Form', BASE_SCRIPT, 'POST');
        
        include_once JAWS_PATH . 'include/Jaws/Widgets/FieldSet.php';
        $fieldset = new Jaws_Widgets_FieldSet(_t('GLOBAL_PROPERTIES'));
        
        //Default page (combo)
        $defaultPage =& Piwi::CreateWidget('Combo', 'default_page');
        $defaultPage->setTitle(_t('STATICPAGE_DEFAULT_PAGE'));
        $pages = $model->getPages();
        if (Jaws_Error::isError($pages)) {
            $pages = array();
        }
        foreach($pages as $page) {
            $defaultPage->addOption($page['title'], $page['base_id']);
        }        
        $defaultPage->setDefault($GLOBALS['app']->Registry->Get('/gadgets/StaticPage/default_page'));
        $fieldset->add($defaultPage);
        
        // Use multilanguage pages?
        $multiLanguage =& Piwi::CreateWidget('Combo', 'multilanguage');
        $multiLanguage->setTitle(_t('STATICPAGE_USE_MULTILANGUAGE'));
        $multiLanguage->addOption(_t('GLOBAL_YES'), 'yes');
        $multiLanguage->addOption(_t('GLOBAL_NO'), 'no');           
        $multiLanguage->setDefault($GLOBALS['app']->Registry->Get('/gadgets/StaticPage/multilanguage'));
        $fieldset->add($multiLanguage);

        // Save Button
        $save =& Piwi::CreateWidget('Button', 'save', _t('GLOBAL_SAVE'), STOCK_SAVE);
        $save->AddEvent(ON_CLICK, 'javascript: updateSettings(this.form);');

        $buttonbox =& Piwi::CreateWidget('HBox');
        $buttonbox->SetStyle(_t('GLOBAL_LANG_DIRECTION')=='rtl'?'float: left;' : 'float: right;');
        $buttonbox->PackStart($save);

        $form->Add($fieldset);
        $form->Add($buttonbox);
        
        $tpl->SetVariable('form', $form->Get());
        $tpl->ParseBlock('Properties');

        return $tpl->Get();
    }

    /**
     * Builds the basic datagrid view
     *
     * @access  private
     * @return  string   XHTML of datagrid
     */
    function DataGrid()
    {
        $model = $GLOBALS['app']->LoadGadget('StaticPage', 'AdminModel');
        $total = $model->TotalOfData('static_pages', 'page_id');

        $grid =& Piwi::CreateWidget('DataGrid', array());
        $grid->TotalRows($total);
        $grid->SetStyle('width: 100%;');
        $grid->SetID('pages_datagrid');
        $grid->useMultipleSelection();
        $grid->AddColumn(Piwi::CreateWidget('Column', _t('GLOBAL_TITLE')));
        $grid->AddColumn(Piwi::CreateWidget('Column', _t('STATICPAGE_FASTURL')));
        $grid->AddColumn(Piwi::CreateWidget('Column', _t('STATICPAGE_PAGE_TRANSLATION')));
        $grid->AddColumn(Piwi::CreateWidget('Column', _t('STATICPAGE_STATUS')));
        $grid->AddColumn(Piwi::CreateWidget('Column', _t('STATICPAGE_LAST_UPDATE')));
        $grid->AddColumn(Piwi::CreateWidget('Column', _t('GLOBAL_ACTIONS')));

        return $grid->Get();
    }

    /**
     * Returns an array with pages found
     *
     * @access  public
     * @param   string  $status  Status of page(s) we want to display
     * @param   string  $search  Keyword (title/description) of pages we want to look for
     * @param   int     $limit   Data limit
     * @return  array   Data
     */
    function GetPages($status, $search, $limit)
    {
        $model = $GLOBALS['app']->LoadGadget('StaticPage', 'AdminModel');
        $pages = $model->SearchPages($status, $search, $limit);
        if (Jaws_Error::IsError($pages)) {
            return array();
        }

        $data    = array();
        $edit_url    = BASE_SCRIPT . '?gadget=StaticPage&amp;action=EditPage&amp;id=';
        $add_turl    = BASE_SCRIPT . '?gadget=StaticPage&amp;action=AddNewTranslation&amp;page=';
        $edit_turl   = BASE_SCRIPT . '?gadget=StaticPage&amp;action=EditTranslation&amp;id=';
        $date = $GLOBALS['app']->loadDate();
        $xss  = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');

        foreach ($pages as $page) {
            $pageData = array();
            $pageData['title'] = '<a href="'.$edit_url.$page['page_id'].'">'.$page['title'].'</a>';

            $param = array('id' => !empty($page['fast_url']) ? $xss->filter($page['fast_url']) : $page['page_id']);
            $furl = $GLOBALS['app']->Map->GetURLFor('StaticPage', 'Page', $param);
            $pageData['furl']  = '<a href="'.$furl.'">'.$page['fast_url'].'</a>';

            $pageData['trans'] = '';
            $pageTranslation   = $model->GetTranslationsOfPage($page['page_id']);
            if (is_array($pageTranslation)) {
                $transString = '';
                $counter     = 0;
                foreach($pageTranslation as $trans) {
                    if ($trans['language'] == $page['base_language']) {
                        continue;
                    }
                    $transString.= '<small><a href="'.$edit_turl.$trans['translation_id'].'">'.$trans['language'].'</a></small>, ';
                    if ($counter % 4 == 0 && $counter != 0) {
                        $transString.= '<br />';
                    }
                    $counter++;
                }
                $pageData['trans'].= substr($transString, 0, -2).'&nbsp;';
            }
            $pageData['trans'].= '<small>[<a href="'.$add_turl.$page['page_id'].'">'._t('STATICPAGE_ADD_LANGUAGE').'</a>]</small>';

            if ($page['published'] === true) {
                $pageData['published'] = _t('STATICPAGE_PUBLISHED');
            } else {
                $pageData['published'] = _t('STATICPAGE_DRAFT');
            }
            $pageData['date']  = $date->Format($page['updated']);
            $actions = '';
            if ($this->GetPermission('EditPage')) {
                $link =& Piwi::CreateWidget('Link', _t('GLOBAL_EDIT'),
                                            $edit_url.$page['page_id'],
                                            STOCK_EDIT);
                $actions.= $link->Get().'&nbsp;';
            }

            if ($this->GetPermission('DeletePage')) {
                $link =& Piwi::CreateWidget('Link', _t('GLOBAL_DELETE'),
                                            "javascript: if (confirm('"._t('GLOBAL_CONFIRM_DELETE', _t('STATICPAGE_PAGE'))."')) ".
                                            "deletePage('".$page['page_id']."');",
                                            STOCK_DELETE);
                $actions.= $link->Get().'&nbsp;';
            }
            $pageData['actions'] = $actions;
            $pageData['__KEY__'] = $page['page_id'];
            $data[] = $pageData;
        }
        return $data;
    }

    /**
     * Display the administration page.
     *
     * @access public
     * @return string
     */
    function Admin()
    {
        $this->CheckPermission('default');
        $this->AjaxMe('script.js');

        $tpl = new Jaws_Template('gadgets/StaticPage/templates/');
        $tpl->Load('AdminStaticPage.html');
        $tpl->SetBlock('static_page');
        
        $tpl->SetVariable('base_script', BASE_SCRIPT);
        $request =& Jaws_Request::getInstance();
        $action  = $request->getRaw('action', 'get');
        $tpl->SetVariable('menubar', $this->MenuBar($action));

        $tpl->SetVariable('grid', $this->DataGrid());

        $toolBar   =& Piwi::CreateWidget('HBox');

        $deleteAll =& Piwi::CreateWidget('Button', 'deleteAllPages',
                                         _t('GLOBAL_DELETE'),
                                         STOCK_DELETE);
        $deleteAll->AddEvent(ON_CLICK,
                             "javascript: massiveDelete('"._t('STATICPAGE_CONFIRM_MASIVE_DELETE_PAGE')."');");

        $toolBar->Add($deleteAll);

        $tpl->SetVariable('tools', $toolBar->Get());
        
        // Language filter
        $settingsModel = $GLOBALS['app']->loadGadget('Settings', 'AdminModel');
        $languages     = $settingsModel->GetLanguageList();
        
        $language = '';
        $languageCombo =& Piwi::CreateWidget('Combo', 'language');
        $languageCombo->setId('language');
        $languageCombo->AddOption('&nbsp;', '');
        foreach($languages as $langCode => $langName) {
            $languageCombo->AddOption($langName, $langCode);
        }
        $languageCombo->SetDefault($language);
        $languageCombo->AddEvent(ON_CHANGE, 'javascript: searchPage();');
        $tpl->SetVariable('language', _t('STATICPAGE_PAGE_LANGUAGE'));
        $tpl->SetVariable('language_field', $languageCombo->Get());

        //Status filter
        $status = '';
        $statusCombo =& Piwi::CreateWidget('Combo', 'status');
        $statusCombo->setId('status');
        $statusCombo->AddOption('&nbsp;', '');
        $statusCombo->AddOption(_t('STATICPAGE_PUBLISHED'), '1');
        $statusCombo->AddOption(_t('STATICPAGE_DRAFT'), '0');
        $statusCombo->SetDefault($status);
        $statusCombo->AddEvent(ON_CHANGE, 'javascript: searchPage();');
        $tpl->SetVariable('status', _t('STATICPAGE_STATUS'));
        $tpl->SetVariable('status_field', $statusCombo->Get());

        // Free text search
        $searchButton =& Piwi::CreateWidget('Button', 'searchButton', _t('GLOBAL_SEARCH'), STOCK_SEARCH);
        $searchButton->AddEvent(ON_CLICK, 'javascript: searchPage();');
        $tpl->SetVariable('search', $searchButton->Get());

        $search = '';
        $searchEntry =& Piwi::CreateWidget('Entry', 'search', $search);
        $searchEntry->SetStyle('zwidth: 100%;');
        $tpl->SetVariable('search_field', $searchEntry->Get());
        
        $tpl->SetVariable('entries', $this->Datagrid());
        $tpl->ParseBlock('static_page');

        return $tpl->Get();
    }

    /**
     * Creates the form to create a translation
     *
     * @access  public
     * @return  string
     */
    function AddNewTranslation()
    {
        $this->CheckPermission('AddPage');

        $model = $GLOBALS['app']->LoadGadget('StaticPage', 'AdminModel');
        //Get Id
        $request =& Jaws_Request::getInstance();
        $page_id = (int)$request->getRaw('page', 'get');

        $page = $model->GetPage($page_id);
        if (Jaws_Error::IsError($page)) {
            $GLOBALS['app']->Session->PushLastResponse($page->GetMessage(), RESPONSE_ERROR);
            require_once JAWS_PATH.'include/Jaws/Header.php';
            Jaws_Header::Location(BASE_SCRIPT . '?gadget=StaticPage');
        }
        return $this->CreateForm($page['title'], '', $page['content'], true, true, '', $page_id,
                                 'AddTranslation', 'translation');
    }

    /**
     * Creates the form to create a new page
     *
     * @access public
     * @return string
     */
    function AddNewPage()
    {
        $this->CheckPermission('AddPage');
        return $this->CreateForm('', '', '', false, true, '', '', 'AddPage');
    }

    /**
     * Save a new page.
     *
     * @access public
     * @return string
     */
    function AddPage()
    {
        $this->CheckPermission('AddPage');
        $model = $GLOBALS['app']->LoadGadget('StaticPage', 'AdminModel');

        $request =& Jaws_Request::getInstance();
        $fetch   = array('title', 'content', 'fast_url', 'language', 'published', 'show_title');
        $post    = $request->getRaw($fetch, 'post');

        $result = $model->AddPage($post['fast_url'], $post['show_title'],
                                  $post['title'], $post['content'], $post['language'],
                                  $post['published']);

        require_once JAWS_PATH . 'include/Jaws/Header.php';
        Jaws_Header::Location(BASE_SCRIPT . '?gadget=StaticPage&action=Admin');
    }

    /**
     * Edit a page.
     *
     * @access public
     * @return string
     */
    function EditPage()
    {
        $this->CheckPermission('EditPage');
        $model = $GLOBALS['app']->LoadGadget('StaticPage', 'AdminModel');

        $request =& Jaws_Request::getInstance();
        $id      = (int)$request->getRaw('id', 'get');

        $page = $model->GetPage($id);
        if (Jaws_Error::IsError($page)) {
            $GLOBALS['app']->Session->PushLastResponse($page->GetMessage(), RESPONSE_ERROR);
            require_once JAWS_PATH.'include/Jaws/Header.php';
            Jaws_Header::Location(BASE_SCRIPT . '?gadget=StaticPage&action=EditPage&id=' . $id);
        }

        return $this->CreateForm($page['title'], $page['fast_url'], $page['content'],
                                 $page['published'], $page['show_title'],  $page['language'],
                                 $id, 'SaveEditPage');
    }

    /**
     * Save an updated page.
     *
     * @access public
     * @return string
     */
    function SaveEditPage()
    {
        $this->CheckPermission('EditPage');
        $model = $GLOBALS['app']->LoadGadget('StaticPage', 'AdminModel');
        $request =& Jaws_Request::getInstance();
        $fetch   = array('page', 'content', 'title', 'fast_url', 'language', 'published', 'show_title');
        $post    = $request->getRaw($fetch, 'post');
        $id      = (int)$post['page'];

        $result = $model->UpdatePage($id, $post['fast_url'], $post['show_title'],
                                     $post['title'], $post['content'], $post['language'], $post['published']);

        require_once JAWS_PATH.'include/Jaws/Header.php';
        Jaws_Header::Location(BASE_SCRIPT . '?gadget=StaticPage&action=EditPage&id=' . $id);
    }
    
    /**
     * Add a new page translation
     *
     * @access  public
     */
    function AddTranslation()
    {
        $this->CheckPermission('EditPage');
        $model = $GLOBALS['app']->LoadGadget('StaticPage', 'AdminModel');
        $request =& Jaws_Request::getInstance();
        $fetch   = array('page', 'title', 'content', 'language', 'published');
        $post    = $request->getRaw($fetch, 'post');
        $page    = (int)$post['page'];
        
        $result = $model->AddTranslation($page, $post['title'], $post['content'], $post['language'], $post['published']);

        require_once JAWS_PATH.'include/Jaws/Header.php';

        if (Jaws_Error::isError($result)) {
            Jaws_Header::Location(BASE_SCRIPT . '?gadget=StaticPage');
        } else {
            $translation = $model->GetPageTranslationByPage($page, $post['language']);
            if (Jaws_Error::isError($translation)) {
                Jaws_Header::Location(BASE_SCRIPT . '?gadget=StaticPage');
            } else {
                Jaws_Header::Location(BASE_SCRIPT . '?gadget=StaticPage&action=EditTranslation&id=' . 
                                     $translation['translation_id']);
            }
        }
    }
    
    /**
     * Creates the form to edit a translation
     *
     * @access  public
     * @return  string
     */
    function EditTranslation()
    {
        $this->CheckPermission('AddPage');

        $model = $GLOBALS['app']->LoadGadget('StaticPage', 'AdminModel');
        //Get Id
        $request  =& Jaws_Request::getInstance();
        $trans_id = (int)$request->getRaw('id', 'get');
        
        $translation = $model->GetPageTranslation($trans_id);
        if (Jaws_Error::IsError($translation)) {
            $GLOBALS['app']->Session->PushLastResponse($translation->GetMessage(), RESPONSE_ERROR);
            require_once JAWS_PATH.'include/Jaws/Header.php';
            Jaws_Header::Location(BASE_SCRIPT . '?gadget=StaticPage');
        }
        return $this->CreateForm($translation['title'], '', $translation['content'], $translation['published'], true, $translation['language'],
                                 $trans_id, 'SaveEditTranslation', 'translation');
    }

    /**
     * Updates a translation
     *
     * @access  public
     * @return  string
     */
    function SaveEditTranslation()
    {
        $this->CheckPermission('EditPage');
        $model = $GLOBALS['app']->LoadGadget('StaticPage', 'AdminModel');
        $request =& Jaws_Request::getInstance();
        $fetch   = array('trans_id', 'title', 'content', 'language', 'published');
        $post    = $request->getRaw($fetch, 'post');
        $trans   = (int)$post['trans_id'];

        $result = $model->UpdateTranslation($trans, $post['title'], $post['content'], $post['language'], $post['published']);

        require_once JAWS_PATH.'include/Jaws/Header.php';
        Jaws_Header::Location(BASE_SCRIPT . '?gadget=StaticPage&action=EditTranslation&id=' . $trans);
    }

    /**
     * Builds a pre-filled form.
     *
     * @access  private
     * @param   string      $title          The title text.
     * @param   string      $fast_url       Fast URL
     * @param   string      $content        The content text.
     * @param   boolean     $published      If the document is published or not
     * @param   boolean     $show_title     If the document should publish the title or not
     * @param   string      $language       Language page has been written
     * @param   int         $id             The ID of the item.
     * @param   string      $action         The action to do when form gets submitted
     * @param   string      $mode           Mode we are using (base by default or translation)
     * @return  string      The completed form.
     */
    function CreateForm($title, $fast_url, $content, $published, $show_title, $language, $id, $action, $mode = 'base')
    {
        $this->AjaxMe('script.js');
        $tpl = new Jaws_Template('gadgets/StaticPage/templates/');
        $tpl->Load('AdminStaticPage.html');
        $tpl->SetBlock('pageform');

        $request =& Jaws_Request::getInstance();
        $m_action  = $request->getRaw('action', 'get');
        $tpl->SetVariable('menubar', $this->MenuBar($m_action));

        $pageform =& Piwi::CreateWidget('Form', BASE_SCRIPT, 'post');

        $pageform->Add(Piwi::CreateWidget('HiddenEntry', 'gadget', 'StaticPage'));
        $pageform->Add(Piwi::CreateWidget('HiddenEntry', 'action', $action));
        if ($action == 'SaveEditTranslation') {
            $pageform->Add(Piwi::CreateWidget('HiddenEntry', 'trans_id', $id));
        } else {
            $pageform->Add(Piwi::CreateWidget('HiddenEntry', 'page', $id));
        }

        $vBox =& Piwi::CreateWidget('VBox');
        $vBox->SetStyle('display: inline;');

        $titleentry =& Piwi::CreateWidget('Entry', 'title', $title);
        $titleentry->SetTitle(_t('GLOBAL_TITLE'));
        $titleentry->SetStyle('width: 300px;');
        $vBox->Add($titleentry);

        if ($mode == 'base') {
            $fasturlentry =& Piwi::CreateWidget('Entry', 'fast_url', $fast_url);
            $fasturlentry->SetTitle(_t('STATICPAGE_FASTURL'));
            $fasturlentry->SetStyle('direction: ltr; width: 300px;');
            $vBox->PackStart($fasturlentry);
        }

        // Status
        $published = ($published === true) ? 1 : 0;
        $statCombo =& Piwi::CreateWidget('Combo', 'published');
        $statCombo->AddOption(_t('STATICPAGE_DRAFT'), '0');
        $statCombo->AddOption(_t('STATICPAGE_PUBLISHED'), '1');
        $statCombo->SetDefault($published);
        $statCombo->setTitle(_t('STATICPAGE_STATUS'));
        $vBox->Add($statCombo);

        if ($mode == 'base') {
            //show title
            $show_title = ($show_title === true) ? 1 : 0;
            $titleShowCombo =& Piwi::CreateWidget('Combo', 'show_title');
            $titleShowCombo->AddOption(_t('GLOBAL_YES'), '1');
            $titleShowCombo->AddOption(_t('GLOBAL_NO'),  '0');
            $titleShowCombo->SetDefault($show_title);
            $titleShowCombo->setTitle(_t('STATICPAGE_SHOW_TITLE'));
            $vBox->Add($titleShowCombo);
        }
        
        $language = (empty($language)) ? $GLOBALS['app']->Registry->Get('/config/site_language') : $language;
        $languageCombo =& Piwi::CreateWidget('Combo', 'language');
        //Load the Settings AdminModel to get a list of all available languages
        $settingsModel = $GLOBALS['app']->loadGadget('Settings', 'AdminModel');
        $languages     = $settingsModel->GetLanguageList();
        foreach($languages as $langCode => $langName) {
            $languageCombo->AddOption($langName, $langCode);
        }
        $languageCombo->SetDefault($language);
        $languageCombo->setTitle(_t('STATICPAGE_PAGE_LANGUAGE'));
        $vBox->Add($languageCombo);
        
        if ($mode == 'base') {
            $autodraft = '<script type="text/javascript" language="javascript">setTimeout(\'startAutoDrafting();\', 1200000);</script>';
            $tpl->SetVariable('autodraft', $autodraft);
        }

        $editor =& $GLOBALS['app']->LoadEditor('StaticPage', 'content', $content, false);
        $editor->TextArea->SetStyle('width: 100%;');
        $editor->SetWidth('750px');

        $pageform->Add($vBox);

        if ($mode == 'base') {
            if ($action == 'AddPage') {
                $submit =& Piwi::CreateWidget('Button', 'newpage', _t('STATICPAGE_ADD_PAGE'), STOCK_SAVE);
            } else {
                $submit =& Piwi::CreateWidget('Button', 'editpage', _t('STATICPAGE_UPDATE_PAGE'), STOCK_SAVE);
            }
        } else {
            if ($action == 'AddTranslation') {
                $submit =& Piwi::CreateWidget('Button', 'newpagetrans', _t('STATICPAGE_ADD_TRANSLATION'), STOCK_SAVE);
            } else {
                $submit =& Piwi::CreateWidget('Button', 'editpagetrans', _t('STATICPAGE_UPDATE_TRANSLATION'), STOCK_SAVE);
            }
        }

        $submit->SetSubmit();

        $cancel =& Piwi::CreateWidget('Button', 'cancel', _t('GLOBAL_CANCEL'), STOCK_CANCEL);
        ///FIXME use the proper url stuff
        $cancel->AddEvent(ON_CLICK, "javascript: window.location = '".BASE_SCRIPT.'?gadget=StaticPage&amp;action=Admin'."';");

        $preview =& Piwi::CreateWidget('Button', 'preview', _t('GLOBAL_PREVIEW'), STOCK_PRINT_PREVIEW);
        $preview->AddEvent(ON_CLICK, 'javascript: parseText(this.form);');

        $buttonbox =& Piwi::CreateWidget('HBox');
        $buttonbox->SetStyle(_t('GLOBAL_LANG_DIRECTION')=='rtl'?'float: left;' : 'float: right;');
        $buttonbox->PackStart($preview);
        $buttonbox->PackStart($cancel);
        $buttonbox->PackStart($submit);

        $pageform->Add($editor);

        $pageform->Add($buttonbox);

        $tpl->setVariable('preview_header', _t('GLOBAL_PREVIEW'));
        $tpl->SetVariable('form', $pageform->Get());

        $tpl->ParseBlock('pageform');

        return $tpl->Get();
    }
}
