<?php
/**
 * StaticPage Gadget
 *
 * @category   GadgetModel
 * @package    StaticPage
 * @author     Jon Wood <jon@jellybob.co.uk>
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2004-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/gpl.html
 */
require_once JAWS_PATH . 'gadgets/StaticPage/Model.php';

class StaticPageAdminModel extends StaticPageModel
{
    /**
     * Install the gadget
     *
     * @access  public
     * @return  boolean  Success/failure
     */
    function InstallGadget()
    {
        $result = $this->installSchema('schema.xml');
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        MDB2::loadFile('Date');
        $variables = array();
        $variables['timestamp'] = MDB2_Date::mdbNow();

        $result = $this->installSchema('insert.xml', $variables, 'schema.xml', true);
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        //registry keys.
        $GLOBALS['app']->Registry->NewKey('/gadgets/StaticPage/hide_title', 'true');
        $GLOBALS['app']->Registry->NewKey('/gadgets/StaticPage/default_page', '1');
        $GLOBALS['app']->Registry->NewKey('/gadgets/StaticPage/multilanguage', 'yes');

        return true;
    }

    /**
     * Uninstalls the gadget
     *
     * @access  public
     * @return  boolean  Success/Failure (Jaws_Error)
     */
    function UninstallGadget()
    {
        $tables = array('static_pages',
                        'static_pages_translation');
        foreach ($tables as $table) {
            $result = $GLOBALS['db']->dropTable($table);
            if (Jaws_Error::IsError($result)) {
                $gName  = _t('STATICPAGE_NAME');
                $errMsg = _t('GLOBAL_ERROR_GADGET_NOT_UNINSTALLED', $gName);
                $GLOBALS['app']->Session->PushLastResponse($errMsg, RESPONSE_ERROR);
                return new Jaws_Error($errMsg, $gName);
            }
        }

        //registry keys
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/StaticPage/hide_title');
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/StaticPage/default_page');
        $GLOBALS['app']->Registry->DeleteKey('/gadgets/StaticPage/multilanguage');

        return true;
    }

    /**
     * Update the gadget
     *
     * @access  public
     * @param   string  $old    Current version (in registry)
     * @param   string  $new    New version (in the $gadgetInfo file)
     * @return  boolean  Success/Failure (Jaws_Error)
     */
    function UpdateGadget($old, $new)
    {
        $result = $this->installSchema('schema.xml', '', "$old.xml");
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        if (version_compare($old, '0.8.0', '<')) {
            $sql = '
                SELECT [page_id], [title], [fast_url], [published], [show_title], [content]
                FROM [[static_page]]';
            $pages = $GLOBALS['db']->queryAll($sql);
            if (Jaws_Error::IsError($pages)) {
                return $pages;
            }

            $site_language = $GLOBALS['app']->Registry->Get('/config/site_language');
            foreach ($pages as $page) {
                $result = $this->AddPage($page['fast_url'], $page['show_title'], $page['title'],
                                         $page['content'], $site_language, $page['published']);
                if (Jaws_Error::IsError($result)) {
                    return $result;
                }
            }

            $result = $GLOBALS['db']->dropTable('static_page');
            if (Jaws_Error::IsError($result)) {
                // do nothing
            }

            // ACL keys
            $GLOBALS['app']->ACL->NewKey('/ACL/gadgets/StaticPage/Properties', 'true');

            // Registry keys.
            $GLOBALS['app']->Registry->NewKey('/gadgets/StaticPage/multilanguage', 'yes');

            $GLOBALS['app']->Session->PopLastResponse(); // emptying all responses message
        }

        return true;
    }

    /**
     * Creates a translation of a given page
     *
     * @access  public
     * @param   string  $page_id    ID of page we are translating
     * @param   string  $title      The translated page title
     * @param   string  $content    The translated page content
     * @param   string  $language   The language we are using
     * @param   boolean $published  If the translated page is published or not
     * @return  boolean Success/failure
     */
    function AddTranslation($page_id, $title, $content, $language, $published)
    {
        //Language exists?
        $language = str_replace(array('.', '/'), '', $language);
        if (!file_exists(JAWS_PATH . "languages/$language/FullName")) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_LANGUAGE_NOT_EXISTS', $language), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_LANGUAGE_NOT_EXISTS', $language), _t('STATICPAGE_NAME'));
        }

        if ($this->TranslationExists($page_id, $language)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_TRANSLATION_EXISTS', $language), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_TRANSLATION_EXISTS', $language), _t('STATICPAGE_NAME'));
        }

        //We already have a translation of this page?
        MDB2::loadFile('Date');
        $params              = array();
        $params['base']      = $page_id;
        $params['title']     = $title;
        $params['content']   = str_replace("\r\n", "\n", $content);
        $params['language']  = $language;
        $params['published'] = (bool)$published;
        $params['now']       = MDB2_Date::mdbNow();

        $sql = '
            INSERT INTO [[static_pages_translation]]
                ([base_id], [title], [content], [language], [published], [updated])
            VALUES
                ({base}, {title}, {content}, {language}, {published}, {now})';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_TRANSLATION_NOT_ADDED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_TRANSLATION_NOT_ADDED'), _t('STATICPAGE_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_TRANSLATION_CREATED'), RESPONSE_NOTICE);
        return true;       
    }

    /**
     * Updates a translation
     *
     * @access  public
     * @param   string  $id         Translation ID
     * @param   string  $title      The translated page title
     * @param   string  $content    The translated page content
     * @param   string  $language   The language we are using
     * @param   boolean $published  If the translated page is published or not
     * @return  boolean Success/failure
     */
    function UpdateTranslation($id, $title, $content, $language, $published)
    {
        //Language exists?
        $language = str_replace(array('.', '/'), '', $language);
        if (!file_exists(JAWS_PATH . "languages/$language/FullName")) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_LANGUAGE_NOT_EXISTS'), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_LANGUAGE_NOT_EXISTS'), _t('STATICPAGE_NAME'));
        }

        //Original language?
        $translation = $this->GetPageTranslation($id);
        if (Jaws_Error::isError($translation)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_TRANSLATION_NOT_EXISTS'), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_TRANSLATION_NOT_EXISTS'), _t('STATICPAGE_NAME'));
        }

        if ($translation['language'] != $language) {
            if ($this->TranslationExists($translation['base_id'], $language)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_TRANSLATION_EXISTS'), RESPONSE_ERROR);
                return new Jaws_Error(_t('STATICPAGE_ERROR_TRANSLATION_EXISTS'), _t('STATICPAGE_NAME'));
            }
        }

        //Lets update it
        MDB2::loadFile('Date');
        $params              = array();
        $params['id']        = $id;
        $params['title']     = $title;
        $params['content']   = str_replace("\r\n", "\n", $content);
        $params['language']  = $language;
        $params['published'] = (bool)$published;
        $params['now']       = MDB2_Date::mdbNow();

        $sql = '
            UPDATE [[static_pages_translation]] SET
              [title]     = {title},
              [content]   = {content},
              [language]  = {language},
              [published] = {published},
              [updated]   = {now}
            WHERE [translation_id] = {id}';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_TRANSLATION_NOT_UPDATED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_TRANSLATION_NOT_UPDATED'), _t('STATICPAGE_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_TRANSLATION_UPDATED'), RESPONSE_NOTICE);
        return true;       
    }

    /**
     * Delete a translation
     *
     * @access  public
     * @param   string  $id         Translation ID
     * @return  boolean Success/failure
     */
    function DeleteTranslation($id)
    {
        $params       = array();
        $params['id'] = $id;
        
        $sql = '
            DELETE FROM [[static_pages_translation]]
            WHERE [translation_id] = {id}';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_TRANSLATION_NOT_DELETED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_TRANSLATION_NOT_DELETED'), _t('STATICPAGE_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_TRANSLATION_DELETED'), RESPONSE_NOTICE);
        return true;       
    }
    

    /**
     * Creates a new page.
     *
     * @access  public
     * @param   string  $fast_url   The fast URL of the page
     * @param   boolean $show_title If the document should publish the title or not
     * @param   string  $title      The title of the page.
     * @param   string  $content    The contents of the page.
     * @param   string  $language   The language of page
     * @param   boolean $published  If the page is published or not
     * @param   boolean $auto       If it's auto saved or not
     * @return  bool    Success/failure
     */
    function AddPage($fast_url, $show_title, $title, $content, $language, $published, $auto = false)
    {
        $fast_url = !empty($fast_url) ? $fast_url : $GLOBALS['app']->UTF8->str_replace(' ', '-', $title);
        if (is_numeric($fast_url)) {
            $fast_url = '-' . $fast_url . '-';
        }
        $fast_url = $GLOBALS['app']->UTF8->str_replace('/', '-', $fast_url);
        $fast_url = $GLOBALS['app']->UTF8->str_replace('\\', '-', $fast_url);

        if ($auto === false) {
            //Current url?
            $fast_url = $this->GetRealFastUrl($fast_url, 'static_pages', 'fast_url');
        }

        $sql = "
            INSERT INTO [[static_pages]]
                ([base_language], [fast_url], [show_title], [updated])
            VALUES
                ({language}, {fast_url}, {show_title}, {now})";

        MDB2::loadFile('Date');
        $params               = array();
        $params['language']   = $language;
        $params['fast_url']   = $fast_url;
        $params['show_title'] = (bool)$show_title;
        $params['now']        = MDB2_Date::mdbNow();

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_PAGE_NOT_ADDED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_PAGE_NOT_ADDED'), _t('STATICPAGE_NAME'));
        }
        $base_id = $GLOBALS['db']->lastInsertID('static_pages', 'page_id');

        $sql = "
            INSERT INTO [[static_pages_translation]]
                ([base_id], [title], [content], [language], [published], [updated])
            VALUES
                ({base_id}, {title}, {content}, {language}, {published}, {now})";

        $params              = array();
        $params['base_id']   = $base_id;
        $params['title']     = $title;
        $params['content']   = str_replace("\r\n", "\n", $content);
        $params['language']  = $language;
        $params['published'] = (bool)$published;
        $params['now']       = MDB2_Date::mdbNow();

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_PAGE_NOT_ADDED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_PAGE_NOT_ADDED'), _t('STATICPAGE_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_PAGE_CREATED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Updates a page.
     *
     * @access  public
     * @param   int     $id             The ID of the page to update.
     * @param   string  $fast_url       The fast URL of the page.
     * @param   boolean $show_title     If the document should publish the title or not
     * @param   string  $title          The new title of the page.
     * @param   string  $content        The new contents of the the page.
     * @param   string  $language       The language of page
     * @param   boolean $published      If the page is published or not
     * @param   boolean $auto           If it's auto saved or not
     * @return  boolean Success/failure
     */
    function UpdatePage($id, $fast_url, $show_title, $title, $content, $language, $published, $auto = false)
    {
        $fast_url = !empty($fast_url) ? $fast_url : $GLOBALS['app']->UTF8->str_replace(' ', '-', $title);
        if (is_numeric($fast_url)) {
            $fast_url = '-' . $fast_url . '-';
        }
        $fast_url = $GLOBALS['app']->UTF8->str_replace('/', '-', $fast_url);
        $fast_url = $GLOBALS['app']->UTF8->str_replace('\\', '-', $fast_url);

        $page = $this->GetPage($id);
        if (Jaws_Error::isError($page)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_PAGE_NOT_FOUND'), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_PAGE_NOT_FOUND'), _t('STATICPAGE_NAME'));
        }

        //Current fast url changes?
        if ($page['fast_url']  != $fast_url && $auto === false) {
            $fast_url           = $this->GetRealFastUrl($fast_url, 'static_pages', 'fast_url');
        }

        $sql = '
            UPDATE [[static_pages]] SET
                [base_language] = {language},
                [fast_url]      = {fast_url},
                [show_title]    = {show_title},
                [updated]       = {now}
            WHERE [page_id] = {id}';

        MDB2::loadFile('Date');
        $params               = array();
        $params['id']         = (int)$id;
        $params['language']   = $language;
        $params['fast_url']   = $fast_url;
        $params['show_title'] = (bool)$show_title;
        $params['now']        = MDB2_Date::mdbNow();

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return new Jaws_Error(_t('STATICPAGE_ERROR_PAGE_NOT_UPDATED'), _t('STATICPAGE_NAME'));
        }

        $sql = '
            UPDATE [[static_pages_translation]] SET
                [title]     = {title},
                [content]   = {content},
                [language]  = {language},
                [published] = {published},
                [updated]   = {now}
            WHERE [translation_id] = {id}';

        $params              = array();
        $params['id']        = $page['translation_id'];
        $params['title']     = $title;
        $params['content']   = str_replace("\r\n", "\n", $content);
        $params['language']  = $language;
        $params['published'] = (bool)$published;
        $params['now']       = MDB2_Date::mdbNow();

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_PAGE_NOT_UPDATED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_PAGE_NOT_UPDATED'), _t('STATICPAGE_NAME'));
        }
        if ($auto) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_PAGE_AUTOUPDATED',
                                                     date('H:i:s'),
                                                     (int)$id,
                                                     date('D, d')),
                                                  RESPONSE_NOTICE);
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_PAGE_UPDATED'), RESPONSE_NOTICE);
        }
        return true;
    }

    /**
     * Deletes a page and all translated of it.
     *
     * @access  public
     * @param   int     $id     The ID of the page to delete.
     * @return  bool    Success/failure
     */
    function DeletePage($id)
    {
        $sql = 'DELETE FROM [[static_pages_translation]] WHERE [base_id] = {id}';
        $result = $GLOBALS['db']->query($sql, array('id' => $id));
        if (Jaws_Error::IsError($result)) {
            return new Jaws_Error(_t('STATICPAGE_ERROR_PAGE_NOT_DELETED'), _t('STATICPAGE_NAME'));
        }

        $sql = 'DELETE FROM [[static_pages]] WHERE [page_id] = {id}';
        $result = $GLOBALS['db']->query($sql, array('id' => $id));
        if (Jaws_Error::IsError($result)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_PAGE_NOT_DELETED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_PAGE_NOT_DELETED'), _t('STATICPAGE_NAME'));
        }

        $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_PAGE_DELETED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Deletes a group of pages
     *
     * @access  public
     * @param   array   $pages  Array with the ids of pages
     * @return  bool    Success/failure
     */
    function MassiveDelete($pages)
    {
        if (!is_array($pages)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_PAGE_NOT_MASSIVE_DELETED'), RESPONSE_ERROR);
            return new Jaws_Error(_t('STATICPAGE_ERROR_PAGE_NOT_MASSIVE_DELETED'), _t('STATICPAGE_NAME'));
        }

        foreach ($pages as $page) {
            $res = $this->DeletePage($page);
            if (Jaws_Error::IsError($res)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_PAGE_NOT_MASSIVE_DELETED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('STATICPAGE_ERROR_PAGE_NOT_MASSIVE_DELETED'), _t('STATICPAGE_NAME'));
            }
        }
        $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_PAGE_MASSIVE_DELETED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Update settings
     *
     * @access  public
     * @param   string  $defaultPage  Default page to use
     * @param   string  $multiLang    Use a multilanguage 'schema'?
     * @return  boolean Success/failure
     */
    function UpdateSettings($defaultPage, $multiLang)
    {
        $res = array();
        $res[0] = $GLOBALS['app']->Registry->Set('/gadgets/StaticPage/default_page', $defaultPage);
        $res[1] = $GLOBALS['app']->Registry->Set('/gadgets/StaticPage/multilanguage', $multiLang);
        
        foreach($res as $r) {
            if (!$r || Jaws_Error::IsError($r)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_ERROR_SETTINGS_NOT_SAVED'), RESPONSE_ERROR);
                return new Jaws_Error(_t('STATICPAGE_ERROR_SETTINGS_NOT_SAVED'), _t('STATICPAGE_NAME'));
            }
        }
        $GLOBALS['app']->Registry->Commit('StaticPage');
        $GLOBALS['app']->Session->PushLastResponse(_t('STATICPAGE_SETTINGS_SAVED'), RESPONSE_NOTICE);
        return true;
    }

    /**
     * Search for pages (and translations) that matches a status and/or a keyword
     * in the title or content
     *
     * @access  public
     * @access  public
     * @param   string  $status  Status of page(s) we want to display
     * @param   string  $search  Keyword (title/description) of pages we want to look for
     * @param   int     $offSet  Data limit
     * @return  array   Array of matches
     */
    function SearchPages($status, $search, $offSet = null)
    {
        $params = array();

        if (!is_bool($status)) {
            if (is_numeric($status)) {
                $params['status'] = $status == 1 ? true : false;
            } elseif (is_string($status)) {
                $params['status'] = $status == 'Y' ? true : false;
            }
        } else {
            $params['status'] = $status;
        }

        $sql = '
            SELECT
                sp.[page_id], spt.[title], sp.[fast_url], sp.[base_language],
                spt.[published], spt.[updated]
            FROM [[static_pages]] sp
            LEFT JOIN [[static_pages_translation]] spt ON sp.[page_id] = spt.[base_id]
            WHERE sp.[base_language] = spt.[language]';

        if (trim($search) != '') {
            $searchdata = explode(' ', $search);
            /**
             * This query needs more work, not use $v straight, should be
             * like rest of the param stuff.
             */
            $i = 0;
            foreach ($searchdata as $v) {
                $v = trim($v);
                $sql .= " AND (spt.[title] LIKE {textLike_".$i."} OR spt.[content] LIKE {textLike_".$i."})";
                $params['textLike_'.$i] = '%'.$v.'%';
                $i++;
            }
        }

        if (trim($status) != '') {
            $sql .= ' AND spt.[published] = {status}';
        }

        if (is_numeric($offSet)) {
            $limit = 10;
            $result = $GLOBALS['db']->setLimit(10, $offSet);
            if (Jaws_Error::IsError($result)) {
                return new Jaws_Error(_t('STATICPAGE_ERROR_PAGES_NOT_RETRIEVED'), _t('STATICPAGE_NAME'));
            }
        }

        $sql.= ' ORDER BY [page_id] ASC';
        $types = array('integer', 'text', 'text', 'text', 'boolean', 'timestamp');
        $result = $GLOBALS['db']->queryAll($sql, $params, $types);
        if (Jaws_Error::IsError($result)) {
            return new Jaws_Error(_t('STATICPAGE_ERROR_PAGES_NOT_RETRIEVED'), _t('STATICPAGE_NAME'));
        }
        //limit, sort, sortDirection, offset..
        return $result;
    }

    /**
     * Implements the GetRealFastURL
     *
     * @access  protected
     * @param   string     $fastUrl   Fast URL
     * @param   string     $table     DB table name (with no [[ ]])
     * @param   string     $field     Table field where fast_url is stored
     * @return  string     Correct fast URL
     */
    function GetRealFastUrl($fastUrl, $table = '', $field = '')
    {
        $params             = array();
        $params['fast_url'] = $fastUrl;

        $sql = '
             SELECT COUNT([page_id])
             FROM [[static_pages]]
             WHERE [fast_url] = {fast_url}';

        $total = $GLOBALS['db']->queryOne($sql, $params);
        if (Jaws_Error::isError($total) || ($total == '0')) {
            return $fastUrl;
        }

        //Get a list of fast_url's that match the current entry
        $params['fast_url'] = $GLOBALS['app']->UTF8->trim($fastUrl).'%';

        $sql = "
             SELECT [fast_url] AS url
             FROM [[static_pages]]
             WHERE [fast_url] LIKE {fast_url}";

        $urlList = $GLOBALS['db']->queryAll($sql, $params);
        if (Jaws_Error::IsError($urlList)) {
            return $fastUrl;
        }

        $counter = 1;
        $numbers = array();
        foreach($urlList as $url) {
            //Matches the foo-\d?
            if (preg_match("/(.+?)-([0-9]{0,})/", $url['url'], $matches)) {
                $numbers[] = (int)$matches[2];
            }
        }
        if (count($numbers) == 0) {
            return $fastUrl . '-1';
        }
        $lastNumber = $numbers[count($numbers)-1];
        $newUrl     = $fastUrl.'-'.($lastNumber+1);
        return $newUrl;
    }
}
