<?php
/**
 * Users AJAX API
 *
 * @category   Ajax
 * @package    Users
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @copyright  2005-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
class UsersAdminAjax extends Jaws_Ajax
{
    /**
     * Constructor
     *
     * @access  public
     */
    function UsersAdminAjax(&$model)
    {
        $this->_Model =& $model;
    }

    /**
     * Get a users's profile
     *
     * @access  public
     * @param   int     $uid  Users's ID
     * @return  array   User's Profile
     */
    function GetUser($uid)
    {
        $this->CheckSession('Users', 'default');
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $model = new Jaws_User();

        $profile = $model->GetInfoById($uid);
        if (Jaws_Error::IsError($profile)) {
            return array();
        }

        require_once JAWS_PATH . 'include/Jaws/Gravatar.php';
        $profile['image'] = Jaws_Gravatar::GetGravatar($profile['email'], 48);
        return $profile;
    }

    /**
     * Get a list of users
     *
     * @access  public
     * @param   boolean $showAll  Show all users (ADMIN and NORMAL) (by default: false)
     * @param   string  $match    Users who match..
     * @return  array   Users list
     */
    function GetUsers($showAll = false, $match = null)
    {
        $this->CheckSession('Users', 'default');
        if (empty($match)) {
            $match = null;
        }

        $type = $GLOBALS['app']->Session->IsSuperAdmin() ? 0 : 1;
        $onlyAdmins = $showAll === true ? false : true;

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $model = new Jaws_User();
        $users = $model->GetAllUsers('name', $match, $type, $onlyAdmins);
        if (count($users) <= 0) {
            return false;
        }

        $types = array(
            '0' => _t('USERS_USERS_TYPE_SUPERADMIN'),
            '1' => _t('USERS_USERS_TYPE_ADMIN'),
            '2' => _t('USERS_USERS_TYPE_NORMAL')
        );

        $list = array();
        foreach ($users as $user) {
            $list[] = array(
                'username' => $user['username'],
                'id'       => $user['id'],
                'realname' => $user['name'],
                'type'     => $types[$user['user_type']],
            );
        }
        return $list;
    }

    /**
     * Adds a new user
     *
     * @access  public
     * @param   string $username  Username
     * @param   string $password  Password
     * @param   string $name      User's name
     * @param   string $email     User's email
     * @param   string $url       User's url
     * @param   int    $guid      Group where user should go
     * @param   string $type      User's type (ADMIN or NORMAL)
     * @return  array  Response (notice or error)
     */
    function AddUser($username, $password, $name, $email, $url, $guid, $type, $enabled)
    {
        $this->CheckSession('Users', 'ManageUsers');

        if ($GLOBALS['app']->Registry->Get('/crypt/enabled') == 'true') {
            require_once JAWS_PATH . 'include/Jaws/Crypt.php';
            $JCrypt = new Jaws_Crypt();
            $JCrypt->Init();
            $password = $JCrypt->rsa->decryptBinary($JCrypt->math->int2bin($password), $JCrypt->pvt_key);
            if (Jaws_Error::isError($password)) {
                $password = '';
            }
        }

        if (trim($username) == '' || trim($name) == '' || trim($password) == '' ||  trim($email) == '') {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_INCOMPLETE_FIELDS'), RESPONSE_ERROR);
            return $GLOBALS['app']->Session->PopLastResponse();
        }

        //Check username
        if (!preg_match('/^[a-z0-9]+$/i', $username) || strlen($username) < 3) {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_REGISTER_USERNAME_NOT_VALID'), RESPONSE_ERROR);
            return $GLOBALS['app']->Session->PopLastResponse();
        }

        if ($type == 0 && $GLOBALS['app']->Session->IsSuperAdmin() === false) {
            $type = 1;
        }

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $model = new Jaws_User();
        $enabled = $enabled == 'true';
        $res = $model->AddUser($username, $name, $email, $url, md5($password), $type, $enabled);
        if ($res === false) {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_NOT_CREATED', $username), RESPONSE_ERROR);
        } else {
            if (isset($guid) && is_numeric($guid) && $type < 2) {
                $model->AddUserToGroup($res, $guid);
            }
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_CREATED', $username), RESPONSE_NOTICE);
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Adds a new group
     *
     * @access  public
     * @param   string $name        Groups's name
     * @param   string $title       Groups's title
     * @param   string $description Groups's description
     * @return  array  Response (notice or error)
     */
    function AddGroup($name, $title, $description)
    {
        $this->CheckSession('Users', 'ManageGroups');
        if (trim($name) == '') {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_GROUPS_INCOMPLETE_FIELDS'), RESPONSE_ERROR);
        } else {
            require_once JAWS_PATH . 'include/Jaws/User.php';
            $model = new Jaws_User();
            $res = $model->AddGroup($name, $title, $description);
            if ($res === false) {
                $GLOBALS['app']->Session->PushLastResponse(_t('USERS_GROUPS_NOT_CREATED', $title), RESPONSE_ERROR);
            } else {
                $GLOBALS['app']->Session->PushLastResponse(_t('USERS_GROUPS_CREATED', $title), RESPONSE_NOTICE);
            }
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Updates an user
     *
     * @access  public
     * @param   int    $uid       User ID
     * @param   string $username  Username
     * @param   string $password  Password
     * @param   string $name      User's name
     * @param   string $email     User's email
     * @param   string $url       User's url
     * @param   string $type      User's type (ADMIN or NORMAL)
     * @return  array  Response (notice or error)
     */
    function UpdateUser($uid, $username, $password, $name, $email, $url, $type, $enabled)
    {
        $this->CheckSession('Users', 'ManageUsers');

        if (trim($username) == '' || trim($name) == '' || trim($email) == '') {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_INCOMPLETE_FIELDS'), RESPONSE_ERROR);
            return $GLOBALS['app']->Session->PopLastResponse();
        }

        if ($GLOBALS['app']->Registry->Get('/crypt/enabled') == 'true') {
            require_once JAWS_PATH . 'include/Jaws/Crypt.php';
            $JCrypt = new Jaws_Crypt();
            $JCrypt->Init();
            $password = $JCrypt->rsa->decryptBinary($JCrypt->math->int2bin($password), $JCrypt->pvt_key);
            if (Jaws_Error::isError($password)) {
                $password = '';
            }
        }

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $model = new Jaws_User();
        $xss   = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        if ($GLOBALS['app']->Session->GetAttribute('user_id') == $uid) {
            $type    = null;
            $enabled = null;
        } else {
            $enabled = $enabled == 'true';
            if (!$GLOBALS['app']->Session->IsSuperAdmin()) {
                $enabled = null;
                $type = ($type == 0)? null : $type;
            }
        }
        $res = $model->UpdateUser($uid,
                                  $xss->parse($username),
                                  $xss->parse($name),
                                  $xss->parse($email),
                                  $xss->parse($url),
                                  md5($password),
                                  $type,
                                  $enabled);
        if ($res === false) {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_NOT_UPDATED', $username), RESPONSE_ERROR);
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_UPDATED', $username), RESPONSE_NOTICE);
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Updates a group
     *
     * @access  public
     * @param   int    $guid        Group ID
     * @param   string $name        Group's name
     * @param   string $title       Groups's title
     * @param   string $description Groups's description
     * @return  array  Response (notice or error)
     */
    function UpdateGroup($guid, $name, $title, $description)
    {
        $this->CheckSession('Users', 'ManageGroups');
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $model = new Jaws_User();
        $xss   = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');

        $res = $model->UpdateGroup($guid, $xss->parse($name), $title, $description);
        if ($res === false) {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_GROUPS_NOT_UPDATED', $title), RESPONSE_ERROR);
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_GROUPS_UPDATED', $title), RESPONSE_NOTICE);
        }

        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Delete an user
     *
     * @access  public
     * @param   int     $uid   User ID
     * @return  array   Response (notice or error)
     */
    function DeleteUser($uid)
    {
        $this->CheckSession('Users', 'ManageUsers');
        $currentUid = $GLOBALS['app']->Session->GetAttribute('user_id');

        if ($currentUid === $uid) {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_CANT_DELETE_SELF'), RESPONSE_ERROR);
        } else {
            require_once JAWS_PATH . 'include/Jaws/User.php';
            $userModel = new Jaws_User();

            $profile = $userModel->GetInfoById($uid);
            if ($GLOBALS['app']->Session->IsSuperAdmin() === false && $profile['user_type'] == 0) {
                $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_CANT_DELETE', $profile['username']), RESPONSE_ERROR);
                return $GLOBALS['app']->Session->PopLastResponse();
            }

            if (!$userModel->DeleteUser($uid)) {
                $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_CANT_DELETE', $profile['username']), RESPONSE_ERROR);
            } else {
                $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USER_DELETED', $profile['username']), RESPONSE_NOTICE);
            }
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Deletes a group
     *
     * @access  public
     * @param   int     $guid   Group ID
     * @return  array   Response (notice or error)
     */
    function DeleteGroup($guid)
    {
        $this->CheckSession('Users', 'ManageGroups');
        $currentUid = $GLOBALS['app']->Session->GetAttribute('user_id');

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();
        $groupinfo = $userModel->GetGroupInfoById($guid);

        if (!$userModel->DeleteGroup($guid)) {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_GROUPS_CANT_DELETE', $groupinfo['name']), RESPONSE_ERROR);
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_GROUPS_DELETED', $groupinfo['name']), RESPONSE_NOTICE);
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Saves only the modified ACL user keys
     *
     * @access  public
     * @param   int     $uid    User' ID
     * @param   array   $keys   ACL Keys
     * @return  array   Response (notice or error)
     */
    function SaveUserACL($uid, $keys)
    {
        $this->CheckSession('Users', 'ManageACL');
        $res = $this->_Model->UpdateUserACL($uid, $keys);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse($res->GetMessage(), RESPONSE_ERROR);
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_ACL_UPDATED'), RESPONSE_NOTICE);
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Resets all ACL keys assigned to an user
     *
     * @access  public
     * @param   int     $uid    User' ID
     * @return  array   Response (notice or error)
     */
    function ResetUserACL($uid)
    {
        $this->CheckSession('Users', 'ManageACL');
        $res = $this->_Model->ResetUserACL($uid);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse($res->GetMessage(), RESPONSE_ERROR);
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_ACL_RESETED'), RESPONSE_NOTICE);
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Save ACL keys of a group
     *
     * @access  public
     * @param   int     $guid   Group ID
     * @param   array   $keys   ACL Keys
     * @return  array   Response (notice or error)
     */
    function SaveGroupACL($guid, $keys)
    {
        $this->CheckSession('Users', 'ManageACL');
        $res = $this->_Model->UpdateGroupACL($guid, $keys);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse($res->GetMessage(), RESPONSE_ERROR);
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_GROUPS_ACL_UPDATED'), RESPONSE_NOTICE);
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Resets all ACL keys assigned to a group
     *
     * @access  public
     * @param   int     $guid   Group ID
     * @return  array   Response (notice or error)
     */
    function ResetGroupACL($guid)
    {
        $this->CheckSession('Users', 'ManageACL');
        $res = $this->_Model->ResetGroupACL($guid);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse($res->GetMessage(), RESPONSE_ERROR);
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_GROUPS_ACL_RESETED'), RESPONSE_NOTICE);
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Add a group of user (by they ids) to a certain group
     *
     * @access  public
     * @param   int     $guid  Group's ID
     * @param   array   $users Array with user id
     * @return  array   Response (notice or error)
     */
    function AddUsersToGroup($guid, $users)
    {
        $this->CheckSession('Users', 'ManageGroups');
        $res = $this->_Model->AddUsersToGroup($guid, $users);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse($res->GetMessage(), RESPONSE_ERROR);
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_GROUPS_UPDATED_USERS'), RESPONSE_NOTICE);
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Save user config settings
     *
     * @access  public
     * @param   string  $priority  Priority
     * @param   string  $method    Authentication method
     * @param   string  $anon      Anonymous users can auto-register
     * @param   integer $type      User's type
     * @param   integer $group     Default group of anonymous registered user
     * @param   string  $recover   Users can recover their passwords
     * @return  array   Response (notice or error)
     */
    function SaveSettings($priority, $method, $anon, $type, $group, $recover)
    {
        $this->CheckSession('Users', 'ManageUsers');
        $res = $this->_Model->SaveSettings($priority, $method, $anon, $type, $group, $recover);
        if (Jaws_Error::IsError($res)) {
            $GLOBALS['app']->Session->PushLastResponse($res->GetMessage(), RESPONSE_ERROR);
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_PROPERTIES_UPDATED'), RESPONSE_NOTICE);
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Returns an array with the ACL keys of a given user
     *
     * @access  public
     * @param   int     $uid    User's ID
     * @param   string  $gadget Gadget's name
     * @return  array   Array with ACL Keys
     */
    function GetUserACLKeys($uid)
    {
        $this->CheckSession('Users', 'ManageACL');
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $model = new Jaws_User();

        $profile = $model->GetInfoById($uid);
        if (isset($profile['username'])) {
            $acl = $this->_Model->GetUserACLKeys($profile['username']);
            return $acl;
        }
        return false;
    }

    /**
     * Returns an array with the ACL keys of a given group
     *
     * @access  public
     * @param   int     $guid   Group's ID
     * @return  array   Array with ACL Keys
     */
    function GetGroupACLKeys($guid)
    {
        $this->CheckSession('Users', 'ManageACL');
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $model = new Jaws_User();

        $profile = $model->GetGroupInfoById($guid);
        if (isset($profile['name'])) {
            $acl = $this->_Model->GetGroupACLKeys($guid);
            return $acl;
        }
        return false;
    }

    /**
     * Get a list of groups
     *
     * @access  public
     * @return  array    Groups list
     */
    function GetGroups()
    {
        $this->CheckSession('Users', 'default');
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $model = new Jaws_User();

        $groups = $model->GetAllGroups('title');
        if (count($groups) <= 0) {
            return null;
        }

        $list = array();
        foreach ($groups as $group) {
            $list[] = array(
                'title' => $group['title'],
                'id'    => $group['id'],
            );
        }
        return $list;
    }

    /**
     * Get the information of a group
     *
     * @access  public
     * @param   int     $guid  Group's ID
     * @return  array   Group's information
     */
    function GetGroup($guid)
    {
        $this->CheckSession('Users', 'default');
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();

        $profile = $userModel->GetGroupInfoById($guid);
        if (Jaws_Error::IsError($profile)) {
            return array();
        }
        return $profile;
    }

    /**
     * Updates my account
     *
     * @access  public
     * @param   string $username  Username
     * @param   string $password  Password
     * @param   string $name      User's name
     * @param   string $email     User's email
     * @param   string $url       User's url
     * @return  array  Response (notice or error)
     */
    function UpdateMyAccount($uid, $username, $password, $name, $email, $url)
    {
        $this->CheckSession('Users', 'default');

        if (trim($username) == '' || trim($name) == '' || trim($email) == '') {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_MYACCOUNT_INCOMPLETE_FIELDS'), RESPONSE_ERROR);
            return $GLOBALS['app']->Session->PopLastResponse();
        }

        if ($GLOBALS['app']->Registry->Get('/crypt/enabled') == 'true') {
            require_once JAWS_PATH . 'include/Jaws/Crypt.php';
            $JCrypt = new Jaws_Crypt();
            $res = $JCrypt->Init();
            if (Jaws_Error::isError($res) || !$res) {
                $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_NOT_UPDATED'), RESPONSE_ERROR);
                return $GLOBALS['app']->Session->PopLastResponse();
            }

            $password = $JCrypt->rsa->decryptBinary($JCrypt->math->int2bin($password), $JCrypt->pvt_key);
            if (empty($password) || Jaws_Error::isError($password)) {
                $password = '';
            }
        }

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();
        $xss       = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');

        $userModel->UpdateUser($uid,
                               $xss->parse($username),
                               $xss->parse($name),
                               $xss->parse($email),
                               $xss->parse($url),
                               md5($password));
        $GLOBALS['app']->Session->PushLastResponse(_t('USERS_MYACCOUNT_UPDATED'), RESPONSE_NOTICE);
        return $GLOBALS['app']->Session->PopLastResponse();
    }

    /**
     * Returns the user form
     *
     * @access  public
     * @return  string  XHTML of userForm
     */
    function GetUserForm()
    {
        $this->CheckSession('Users', 'default');
        $gadget = $GLOBALS['app']->LoadGadget('Users', 'AdminHTML');
        return $gadget->EditUser();
    }

    /**
     * Returns the group form
     *
     * @access  public
     * @return  string  XHTML of groupForm
     */
    function GetGroupForm()
    {
        $this->CheckSession('Users', 'default');
        $gadget = $GLOBALS['app']->LoadGadget('Users', 'AdminHTML');
        return $gadget->EditGroup();
    }

    /**
     * Get the user-group form
     *
     * @access  public
     * @param   int     $guid    Group ID
     * @return  string
     */
    function GetUserGroupsForm($guid)
    {
        $this->CheckSession('Users', 'default');
        $gadget = $GLOBALS['app']->LoadGadget('Users', 'AdminHTML');
        return $gadget->GetUserGroupUI($guid);
    }
    
    /**
     * Returns the UI for the advanced options for users
     *
     * @access  public
     * @param   int     $uid     User ID
     * @return  string
     */
    function GetAdvUserOptionsUI($uid)
    {
        $this->CheckSession('Users', 'default');
        $gadget = $GLOBALS['app']->LoadGadget('Users', 'AdminHTML');
        return $gadget->GetAdvancedUserOptionsUI($uid);
    }
    
    /**
     * Save advanced options of selected user
     *
     * @access  public
     * @param   int     $uid     User ID
     * @param   string  $lang    User language
     * @param   string  $theme   User theme
     * @param   string  $editor  User editor
     * @return  array  Response (notice or error)
     */
    function SaveAdvancedOptions($uid, $lang, $theme, $editor, $timezone)
    {
        $this->CheckSession('Users', 'default');

        if ($lang == '-default-') {
            $lang = null;
        }

        if ($theme == '-default-') {
            $theme = null;
        }

        if ($editor == '-default-') {
            $editor = null;
        }

        if ($timezone == '-default-') {
            $timezone = null;
        }

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $model = new Jaws_User();
        $res = $model->UpdateAdvancedOptions($uid, array(
                                                         'language' => $lang, 
                                                         'theme'    => $theme,
                                                         'editor'   => $editor,
                                                         'timezone' => $timezone,
                                                         )
                                             );
        if ($res === false) {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_NOT_ADVANCED_UPDATED'), RESPONSE_ERROR);
        } else {
            $GLOBALS['app']->Session->PushLastResponse(_t('USERS_USERS_ADVANCED_UPDATED'), RESPONSE_NOTICE);
        }
        return $GLOBALS['app']->Session->PopLastResponse();
    }

}
