<?php
/**
 * Users Core Gadget Admin
 *
 * @category   GadgetAdmin
 * @package    Users
 * @author     Jonathan Hernandez <ion@suavizado.com>
 * @copyright  2004-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
class UsersAdminHTML extends Jaws_GadgetHTML
{
    /**
     * Gadget constructor
     *
     * @access public
     */
    function UsersAdminHTML()
    {
        $this->Init('Users');
    }

    /**
     * Calls default admin action
     *
     * @access public
     * @return string  Template content
     */
    function Admin()
    {
        $this->CheckPermission('default');
        if ($this->GetPermission('ManageUsers')) {
            return $this->UsersView();
        } elseif ($this->GetPermission('ManageGroups')) {
            return $this->Groups();
        }

        $this->CheckPermission('ManageACL');
    }

    /**
     * Prepares the users menubar
     *
     * @access  public
     * @param   string  $action   Selected action
     * @return  string  XHTML of menubar
     */
    function MenuBar($action)
    {
        $actions = array('Users', 'Groups', 'Properties');
        if (!in_array($action, $actions)) {
            $action = 'Users';
        }

        require_once JAWS_PATH . 'include/Jaws/Widgets/Menubar.php';
        $menubar = new Jaws_Widgets_Menubar();
        if ($this->GetPermission('ManageUsers')) {
            $menubar->AddOption('Users', _t('USERS_NAME'),
                                BASE_SCRIPT . '?gadget=Users&amp;action=Admin', 'gadgets/Users/images/users_mini.png');
        }
        if ($this->GetPermission('ManageGroups')) {
            $menubar->AddOption('Groups', _t('USERS_GROUPS_GROUPS'),
                                BASE_SCRIPT . '?gadget=Users&amp;action=Groups', 'gadgets/Users/images/groups_mini.png');
        }
        $menubar->AddOption('Properties', _t('GLOBAL_PROPERTIES'),
                            BASE_SCRIPT . '?gadget=Users&amp;action=Properties', STOCK_PREFERENCES);
        $menubar->Activate($action);
        return $menubar->Get();
    }

    /**
     * Show user administration
     *
     * @access  public
     * @return  string HTML content
     */
    function UsersView()
    {
        $request =& Jaws_Request::getInstance();
        $get = $request->get(array('action', 'id'), 'get');
        $action = !is_null($get['action']) ? $get['action'] : 'Admin';
        if ($action != 'MyAccount') {
            $this->CheckPermission('ManageUsers');
        }
        $this->AjaxMe('script.js');
        $GLOBALS['app']->Layout->AddScriptLink('libraries/xtree/xtree.js');
        if ($GLOBALS['app']->Registry->Get('/crypt/enabled') == 'true') {
            $GLOBALS['app']->Layout->AddScriptLink('libraries/js/bigint.js');
            $GLOBALS['app']->Layout->AddScriptLink('libraries/js/bigintmath.js');
            $GLOBALS['app']->Layout->AddScriptLink('libraries/js/rsa.js');
        }

        $tpl = new Jaws_Template('gadgets/Users/templates/');
        $tpl->Load('Users.html');
        $tpl->SetBlock('Users');

        // Right menu
        include_once JAWS_PATH . 'include/Jaws/Widgets/XHTMLMenu.php';
        $menu = new Jaws_Widgets_XHTMLMenu('', 'right_menu', 'visibility: hidden;');

        $addUser =& Piwi::CreateWidget('Button', 'add_user', _t('USERS_USERS_ADD'), 'gadgets/Users/images/add_user.png');
        $addUser->AddEvent(ON_CLICK, "javascript: addUser();");
        $tpl->SetVariable('add_user', $addUser->Get());

        $saveUser =& Piwi::CreateWidget('Button', 'save_user', _t('GLOBAL_SAVE'), STOCK_SAVE);
        $saveUser->AddEvent(ON_CLICK, "javascript: saveUser();");
        $saveUser->SetStyle('display: none;');
        $tpl->SetVariable('save_user', $saveUser->Get());

        $cancelAction =& Piwi::CreateWidget('Button', 'cancel_action', _t('GLOBAL_CANCEL'), STOCK_CANCEL);
        $cancelAction->AddEvent(ON_CLICK, "javascript: stopAction();");
        $cancelAction->SetStyle('display: none;');
        $tpl->SetVariable('cancel', $cancelAction->Get());

        if ($this->GetPermission('ManageACL')) {
            $resetACL =& Piwi::CreateWidget('Button', 'reset_acl', _t('USERS_RESET_ACL'), STOCK_UNDO);
            $resetACL->AddEvent(ON_CLICK, "javascript: resetUserACL();");
            $resetACL->SetStyle('display: none;');
            $tpl->SetVariable('reset_acl', $resetACL->Get());

            $saveACL =& Piwi::CreateWidget('Button', 'save_acl', _t('GLOBAL_SAVE'), STOCK_SAVE);
            $saveACL->AddEvent(ON_CLICK, "javascript: saveACL();");
            $saveACL->SetStyle('display: none;');
            $tpl->SetVariable('save_acl', $saveACL->Get());

            $menu->addOption('manage_acl', _t('USERS_ACLRULES'), "javascript: manageUserACL();", 'gadgets/Users/images/acls.png');
        }

        $menu->addOption('delete_user', _t('USERS_ACCOUNT_DELETE'), "javascript: deleteUser();", STOCK_DELETE, false, '', true);
        //Advanced options, such as language.. theme
        $menu->addOption('advanced_options', _t('USERS_ADVANCED_USER_EDIT'), "javascript: advancedUserOptions();",
                         STOCK_FONT, false, '', true);
        $tpl->SetVariable('right_menu', $menu->get());
        
        //search tool
        $searchTool =& Piwi::CreateWidget('HBox');
        $searchTool->SetID('users_search');

        $searchField  =& Piwi::CreateWidget('Entry', 'user_search_entry', '');
        $searchButton =& Piwi::CreateWidget('Button', 'user_search_button', _t('USERS_FIND_USER'), STOCK_SEARCH);
        $searchButton->AddEvent(ON_CLICK, 'javascript: searchUser();');

        $showAll =& Piwi::CreateWidget('CheckButtons', 'show_all');
        $showAll->AddOption(_t('USERS_USERS_SHOW_ALL'), 'ALL', 'show_all', true);
        $showAll->AddEvent(ON_CHANGE, 'javascript: getUsers();');

        //$searchTool->Add($searchField);
        //$searchTool->Add($searchButton);
        $searchTool->Add($showAll);
        $tpl->SetVariable('searchTool_users', $searchTool->Get());

        //Menu bar
        $tpl->SetVariable('menubar', $this->MenuBar('Users'));

        //Fill the users combo..
        $comboUsers =& Piwi::CreateWidget('Combo', 'users_combo');
        $comboUsers->SetSize(20);
        $comboUsers->SetStyle('width: 200px; height: 350px;');
        $comboUsers->AddEvent(ON_CHANGE, 'javascript: editUser(this.value);');
        //Which users should we fetch first-time?
        $typeOfUsers = $GLOBALS['app']->Session->IsSuperAdmin() ? 0 : 1;
        $onlyAdmins = false;

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();
        $users = $userModel->GetAllUsers('name', null, $typeOfUsers, $onlyAdmins);
        foreach($users as $user) {
            $comboUsers->AddOption($user['username'], $user['id']);
        }

        $tpl->SetVariable('combo_users', $comboUsers->Get());
        $tpl->SetVariable('noGroup', _t('USERS_GROUPS_NOGROUP'));
        $tpl->SetVariable('wrongPassword', _t('USERS_USERS_PASSWORDS_DONT_MATCH'));
        $tpl->SetVariable('incompleteUserFields', _t('USERS_USERS_INCOMPLETE_FIELDS'));
        $tpl->SetVariable('permissionsMsg', _t('USERS_USERS_PERMISSIONS'));
        $tpl->SetVariable('selectUser', _t('USERS_USERS_SELECT_A_USER'));
        $tpl->SetVariable('confirmUserDelete', _t('USERS_USER_CONFIRM_DELETE'));
        $tpl->SetVariable('confirmResetACL', _t('USERS_RESET_ACL_CONFIRM'));
        $tpl->ParseBlock('Users');

        return $tpl->Get();
    }

    /**
     * Prepares the group management view
     *
     * @access  public
     * @return  string  XHTML of view
     */
    function Groups()
    {
        $this->AjaxMe('script.js');
        $GLOBALS['app']->Layout->AddScriptLink('libraries/xtree/xtree.js');
        $tpl = new Jaws_Template('gadgets/Users/templates/');
        $tpl->Load('Groups.html');
        $tpl->SetBlock('Groups');

        // Right menu
        include_once JAWS_PATH . 'include/Jaws/Widgets/XHTMLMenu.php';
        $menu = new Jaws_Widgets_XHTMLMenu('', 'right_menu', 'visibility: hidden;');

        $addGroup =& Piwi::CreateWidget('Button', 'add_group', _t('USERS_GROUPS_ADD'), 'gadgets/Users/images/add_group.png');
        $addGroup->AddEvent(ON_CLICK, "javascript: addGroup();");
        $tpl->SetVariable('add_group', $addGroup->Get());

        $saveGroup =& Piwi::CreateWidget('Button', 'save_group', _t('GLOBAL_SAVE'), STOCK_SAVE);
        $saveGroup->AddEvent(ON_CLICK, "javascript: saveGroup();");
        $saveGroup->SetStyle('display: none;');
        $tpl->SetVariable('save_group', $saveGroup->Get());

        $cancelAction =& Piwi::CreateWidget('Button', 'cancel_action', _t('GLOBAL_CANCEL'), STOCK_CANCEL);
        $cancelAction->AddEvent(ON_CLICK, "javascript: stopAction();");
        $cancelAction->SetStyle('display: none;');
        $tpl->SetVariable('cancel', $cancelAction->Get());

        $menu->addOption('add_usergroups', _t('USERS_GROUPS_ADD_USERS'), "javascript: addUsersToGroup();", 'gadgets/Users/images/add_group.png');

        if ($this->GetPermission('ManageACL')) {
            $resetACL =& Piwi::CreateWidget('Button', 'reset_acl', _t('USERS_RESET_ACL'), STOCK_UNDO);
            $resetACL->AddEvent(ON_CLICK, "javascript: resetGroupACL();");
            $resetACL->SetStyle('display: none;');
            $tpl->SetVariable('reset_acl', $resetACL->Get());

            $saveACL =& Piwi::CreateWidget('Button', 'save_acl', _t('GLOBAL_SAVE'), STOCK_SAVE);
            $saveACL->AddEvent(ON_CLICK, "javascript: saveGroupACL();");
            $saveACL->SetStyle('display: none;');
            $tpl->SetVariable('save_acl', $saveACL->Get());

            $menu->addOption('manage_acl', _t('USERS_ACLRULES'), "javascript: manageGroupACL();", 'gadgets/Users/images/acls.png');
        }

        $menu->addOption('delete_group', _t('USERS_GROUPS_DELETE'), "javascript: deleteGroup();", STOCK_DELETE, false, '', true);
        $tpl->SetVariable('right_menu', $menu->get());

        //Menu bar
        $tpl->SetVariable('menubar', $this->MenuBar('Groups'));

        //Fill the groups combo..
        $comboGroups =& Piwi::CreateWidget('Combo', 'groups_combo');
        $comboGroups->SetSize(20);
        $comboGroups->SetStyle('width: 200px; height: 350px;');
        $comboGroups->AddEvent(ON_CHANGE, 'javascript: editGroup(this.value);');

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();
        $groups = $userModel->GetAllGroups('title');
        foreach ($groups as $group) {
            $comboGroups->AddOption($group['title'], $group['id']);
        }

        $tpl->SetVariable('combo_groups', $comboGroups->Get());
        $tpl->SetVariable('incompleteGroupFields', _t('USERS_GROUPS_INCOMPLETE_FIELDS'));
        $tpl->SetVariable('permissionsMsg', _t('USERS_USERS_PERMISSIONS'));
        $tpl->SetVariable('confirmGroupDelete', _t('USERS_GROUPS_CONFIRM_DELETE'));
        $tpl->SetVariable('confirmResetACL', _t('USERS_RESET_ACL_CONFIRM'));
        $tpl->ParseBlock('Groups');

        return $tpl->Get();
    }

    /**
     * Show account settings for logged user
     *
     * @access public
     * @return string  Template content
     */
    function MyAccount()
    {
        require_once JAWS_PATH.'include/Jaws/User.php';
        $username = $GLOBALS['app']->Session->GetAttribute('username');
        $userModel = new Jaws_User();
        $userInfo = $userModel->GetInfoByUsername($username);

        $this->AjaxMe('script.js');
        $use_crypt = ($GLOBALS['app']->Registry->Get('/crypt/enabled') == 'true')? true : false;
        if ($use_crypt) {
            require_once JAWS_PATH . 'include/Jaws/Crypt.php';
            $JCrypt = new Jaws_Crypt();
            $use_crypt = $JCrypt->Init();
        }

        if ($use_crypt) {
            $GLOBALS['app']->Layout->AddScriptLink('libraries/js/bigint.js');
            $GLOBALS['app']->Layout->AddScriptLink('libraries/js/bigintmath.js');
            $GLOBALS['app']->Layout->AddScriptLink('libraries/js/rsa.js');
        }

        $tpl = new Jaws_Template('gadgets/Users/templates/');
        $tpl->Load('MyAccount.html');
        $tpl->SetBlock('MyAccount');
        if (!Jaws_Error::IsError($userInfo) && isset($userInfo['username'])) {
            $tpl->SetVariable('incompleteUserFields', _t('USERS_MYACCOUNT_INCOMPLETE_FIELDS'));
            $tpl->SetVariable('wrongPassword', _t('USERS_MYACCOUNT_PASSWORDS_DONT_MATCH'));

            $form =& Piwi::CreateWidget('Form', BASE_SCRIPT, 'post');
            $form->Add(Piwi::CreateWidget('HiddenEntry', 'gadget', 'Users'));
            $form->Add(Piwi::CreateWidget('HiddenEntry', 'action', 'UpdateMyAccount'));
            $form->Add(Piwi::CreateWidget('HiddenEntry', 'id', $userInfo['id']));

            if ($use_crypt) {
                $form->Add(Piwi::CreateWidget('HiddenEntry', 'modulus',  $JCrypt->math->bin2int($JCrypt->pub_key->getModulus())));
                $form->Add(Piwi::CreateWidget('HiddenEntry', 'exponent', $JCrypt->math->bin2int($JCrypt->pub_key->getExponent())));
            }

            include_once JAWS_PATH . 'include/Jaws/Widgets/FieldSet.php';
            $fieldset = new Jaws_Widgets_FieldSet(_t('USERS_USERS_ACCOUNT_INFO'));
            $fieldset->SetDirection('vertical');

            $hbox =& Piwi::CreateWidget('HBox');
            $hbox->setStyle('width: 100%;');
            $hbox->_useTitles = true;

            $usernameEntry =& Piwi::CreateWidget('Entry', 'username', $userInfo['username']);
            $usernameEntry->SetTitle(_t('USERS_USERS_USERNAME'));
            $hbox->packstart($usernameEntry);
            $avatar =& Piwi::CreateWidget('Image',
                                          $userModel->GetAvatar($userInfo['username'], $userInfo['email']), $userInfo['username']);
            $avatar->SetID('user_avatar');
            $avatar->setStyle('float: right;');
            $hbox->packstart($avatar);

            $fieldset->Add($hbox);

            $nameEntry =& Piwi::CreateWidget('Entry', 'name', $userInfo['name']);
            $nameEntry->SetTitle(_t('GLOBAL_NAME'));
            $nameEntry->SetStyle('width: 250px');
            $fieldset->Add($nameEntry);

            $emailEntry =& Piwi::CreateWidget('Entry', 'email', $userInfo['email']);
            $emailEntry->SetTitle(_t('GLOBAL_EMAIL'));
            $emailEntry->SetStyle('width: 250px');
            $fieldset->Add($emailEntry);

            $urlEntry =& Piwi::CreateWidget('Entry', 'url', $userInfo['url']);
            $urlEntry->SetTitle(_t('GLOBAL_URL'));
            $urlEntry->SetStyle('width: 250px');
            $fieldset->Add($urlEntry);

            $pass1Entry =& Piwi::CreateWidget('PasswordEntry', 'pass1', '');
            $pass1Entry->SetTitle(_t('USERS_USERS_PASSWORD'));
            $fieldset->Add($pass1Entry);
            $pass2Entry =& Piwi::CreateWidget('PasswordEntry', 'pass2', '');
            $pass2Entry->SetTitle(_t('USERS_USERS_PASSWORD_VERIFY'));
            $fieldset->Add($pass2Entry);

            $form->Add($fieldset);

            $buttonbox =& Piwi::CreateWidget('HBox');
            $buttonbox->SetStyle('float: right;'); //hig style
            $submit =& Piwi::CreateWidget('Button', 'SubmitButton', _t('GLOBAL_UPDATE'), STOCK_SAVE);
            $submit->AddEvent(ON_CLICK, "javascript: updateMyAccount();");
            $buttonbox->Add($submit);

            $form->Add($buttonbox);

            $tpl->SetVariable('form', $form->Get());
        }
        $tpl->ParseBlock('MyAccount');

        return $tpl->Get();
    }

    /**
     * Show a form to edit a given user
     *
     * @access  public
     * @return  string HTML content
     */
    function EditUser()
    {
        $form =& Piwi::CreateWidget('Form', BASE_SCRIPT, 'post', '', 'userInfo_form');
        $form->Add(Piwi::CreateWidget('HiddenEntry', 'userInfo_action', 'SaveEditUser'));
        $form->Add(Piwi::CreateWidget('HiddenEntry', 'userInfo_uid', ''));
        $form->SetStyle('padding: 0px;');

        $use_crypt = ($GLOBALS['app']->Registry->Get('/crypt/enabled') == 'true')? true : false;
        if ($use_crypt) {
            require_once JAWS_PATH . 'include/Jaws/Crypt.php';
            $JCrypt = new Jaws_Crypt();
            $use_crypt = $JCrypt->Init();
        }
        if ($use_crypt) {
            $form->Add(Piwi::CreateWidget('HiddenEntry', 'modulus',  $JCrypt->math->bin2int($JCrypt->pub_key->getModulus())));
            $form->Add(Piwi::CreateWidget('HiddenEntry', 'exponent', $JCrypt->math->bin2int($JCrypt->pub_key->getExponent())));
        }

        include_once JAWS_PATH . 'include/Jaws/Widgets/FieldSet.php';
        $fieldset = new Jaws_Widgets_FieldSet();
        $fieldset->SetDirection('vertical');
        $fieldset->SetID('usersFieldset');
        $fieldset->SetStyle('border: 0px; padding: 0px; margin: 0px;');

        $hbox =& Piwi::CreateWidget('HBox');
        $hbox->setStyle((_t('GLOBAL_LANG_DIRECTION')=='rtl'?'text-align: right;' : 'text-align: left;').' width: 100%;');
        $hbox->_useTitles = true;

        $usernameEntry =& Piwi::CreateWidget('Entry', 'username', '');
        $usernameEntry->SetTitle(_t('USERS_USERS_USERNAME'));
        $hbox->packstart($usernameEntry);

        $avatar =& Piwi::CreateWidget('Image', 'gadgets/Users/images/no-photo.png',
                                      '', 0, 48, 48);
        $avatar->setStyle('float: right;');
        $avatar->SetID('user_avatar');
        $hbox->packstart($avatar);
        $fieldset->Add($hbox);

        $nameEntry =& Piwi::CreateWidget('Entry', 'name', '');
        $nameEntry->SetTitle(_t('GLOBAL_NAME'));
        $nameEntry->SetStyle('width: 250px');
        $fieldset->Add($nameEntry);

        $emailEntry =& Piwi::CreateWidget('Entry', 'email', '');
        $emailEntry->SetTitle(_t('GLOBAL_EMAIL'));
        $emailEntry->SetStyle('width: 250px; direction: ltr;');
        $fieldset->Add($emailEntry);

        $urlEntry =& Piwi::CreateWidget('Entry', 'url', '');
        $urlEntry->SetTitle(_t('GLOBAL_URL'));
        $urlEntry->SetStyle('width: 250px; direction: ltr;');
        $fieldset->Add($urlEntry);

        $select =& Piwi::createWidget('Combo', 'user_group', _t('USERS_GROUPS_ADD_USER'));
        $select->SetID('user_group');
        $select->addOption('&nbsp;', '');
        $select->setDefault('');
        $fieldset->add($select);

        $userType =& Piwi::CreateWidget('Combo', 'user_type', _t('USERS_USERS_TYPE'));
        $userType->SetID('user_type');
        if ($GLOBALS['app']->Session->IsSuperAdmin() === true) {
            $userType->AddOption(_t('USERS_USERS_TYPE_SUPERADMIN'), '0');
        }
        $userType->AddOption(_t('USERS_USERS_TYPE_ADMIN'), '1');
        $userType->AddOption(_t('USERS_USERS_TYPE_NORMAL'), '2');
        $fieldset->Add($userType);

        $pass1Entry =& Piwi::CreateWidget('PasswordEntry', 'pass1', '');
        $pass1Entry->SetTitle(_t('USERS_USERS_PASSWORD'));
        $pass1Entry->SetStyle('direction: ltr;');
        $fieldset->Add($pass1Entry);

        $pass2Entry =& Piwi::CreateWidget('PasswordEntry', 'pass2', '');
        $pass2Entry->SetTitle(_t('USERS_USERS_PASSWORD_VERIFY'));
        $pass2Entry->SetStyle('direction: ltr;');
        $fieldset->Add($pass2Entry);

        $enabled =& Piwi::CreateWidget('Combo', 'enabled');
        $enabled->SetTitle(_t('GLOBAL_ENABLED'));
        $enabled->SetID('enabled');
        $enabled->AddOption(_t('GLOBAL_NO'),  'false');
        $enabled->AddOption(_t('GLOBAL_YES'), 'true');
        $enabled->SetDefault('true');
        $fieldset->Add($enabled);

        $form->Add($fieldset);

        return $form->Get();
    }

    /**
     * Show a form to edit a given group
     *
     * @access  public
     * @return  string HTML content
     */
    function EditGroup()
    {
        $tpl = new Jaws_Template('gadgets/Users/templates/');
        $tpl->Load('GroupInfo.html');
        $tpl->SetBlock('GroupInfo');

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();

        $form =& Piwi::CreateWidget('Form', BASE_SCRIPT, 'post', '', 'groupInfo_form');
        $form->SetStyle('padding: 0px;');

        include_once JAWS_PATH . 'include/Jaws/Widgets/FieldSet.php';
        $fieldset = new Jaws_Widgets_FieldSet();
        $fieldset->SetDirection('vertical');
        $fieldset->SetStyle('border: 0px; padding: 0px; margin: 0px; height: 1px;');

        $nameEntry =& Piwi::CreateWidget('Entry', 'name', '');
        $nameEntry->SetTitle(_t('GLOBAL_NAME'));
        $nameEntry->SetStyle('width: 250px;');

        $titleEntry =& Piwi::CreateWidget('Entry', 'title', '');
        $titleEntry->SetTitle(_t('GLOBAL_TITLE'));
        $titleEntry->SetStyle('width: 250px;');

        $descEntry =& Piwi::CreateWidget('TextArea', 'description', '');
        $descEntry->SetTitle(_t('GLOBAL_DESCRIPTION'));
        $descEntry->SetStyle('width: 250px;');
        $descEntry->SetRows(5);
        $descEntry->SetColumns(60);

        $fieldset->Add($nameEntry);
        $fieldset->Add($titleEntry);
        $fieldset->Add($descEntry);
        $form->Add($fieldset);

        $tpl->SetVariable('form', $form->Get());
        $tpl->ParseBlock('GroupInfo');
        return $tpl->Get();
    }

    /**
     * Returns a mini-form to edit advanced options of a selected user
     *
     * @access  public
     * @param   int     $uid    User ID
     * @return  string
     */
    function GetAdvancedUserOptionsUI($uid)
    {
        $tpl = new Jaws_Template('gadgets/Users/templates/');
        $tpl->Load('AdvancedUserOptions.html');
        $tpl->SetBlock('AdvancedUserOptions');
        
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();

        //Load the user info
        $info = $userModel->GetInfoById($uid);
        if (is_null($info['theme'])) {
            $info['theme'] = '-default-';
        }

        if (is_null($info['language'])) {
            $info['language'] = '-default-';
        }

        if (is_null($info['editor'])) {
            $info['editor'] = '-default-';
        }

        $form =& Piwi::CreateWidget('Form', BASE_SCRIPT, 'post', '', 'advanceduserOpts_form');
        $form->SetStyle('padding: 0px;');

        include_once JAWS_PATH . 'include/Jaws/Widgets/FieldSet.php';
        $fieldset = new Jaws_Widgets_FieldSet();
        $fieldset->SetDirection('vertical');
        $fieldset->SetStyle('border: 0px; padding: 0px; margin: 0px;');

        //Here we load the Settings/Layout models (which is part of core) to extract some data
        $settingsModel = $GLOBALS['app']->loadGadget('Settings', 'AdminModel');
        $layoutModel   = $GLOBALS['app']->loadGadget('Layout', 'AdminModel');

        //Language
        $lang =& Piwi::CreateWidget('Combo', 'user_language');
        $lang->setID('user_language');
        $lang->AddOption(_t('USERS_ADVANCED_OPTS_NOT_YET'), '-default-');
        $languages = $settingsModel->GetLanguageList();
        foreach($languages as $k => $v) {
            $lang->AddOption($v, $k);
        }
        $lang->SetDefault($info['language']);
        $lang->SetTitle(_t('USERS_ADVANCED_OPTS_LANGUAGE'));
        $fieldset->Add($lang);

        //Theme
        $theme =& Piwi::CreateWidget('Combo', 'user_theme');
        $theme->setID('user_theme');
        $theme->AddOption(_t('USERS_ADVANCED_OPTS_NOT_YET'), '-default-');
        $themes = $layoutModel->GetThemeList();
        foreach($themes as $k => $v) {
            $theme->AddOption($v, $v);
        }
        $theme->SetDefault($info['theme']);
        $theme->SetTitle(_t('USERS_ADVANCED_OPTS_THEME'));
        $fieldset->Add($theme);

        //Editor
        $GLOBALS['app']->Translate->loadTranslation('Settings', JAWS_PATH . 'gadgets');        
        $editor =& Piwi::CreateWidget('Combo', 'user_editor');
        $editor->setID('user_editor');
        $editor->AddOption(_t('USERS_ADVANCED_OPTS_NOT_YET'), '-default-');
        $editors = $settingsModel->GetEditorList();
        foreach($editors as $k => $v) {
            $editor->AddOption($v, $k);
        }
        $editor->SetDefault($info['editor']);
        $editor->SetTitle(_t('USERS_ADVANCED_OPTS_EDITOR'));
        $fieldset->Add($editor);

        //Time Zones
        $timezone =& Piwi::CreateWidget('Combo', 'user_timezone');
        $timezone->setID('user_timezone');
        $timezone->AddOption(_t('USERS_ADVANCED_OPTS_NOT_YET'), '-default-');
        $timezones = $settingsModel->GetTimeZonesList();
        foreach($timezones as $k => $v) {
            $timezone->AddOption($v, $k);
        }
        $timezone->SetDefault($info['timezone']);
        $timezone->SetTitle(_t('GLOBAL_TIMEZONE'));
        $fieldset->Add($timezone);

        $form->add($fieldset);
        $tpl->SetVariable('form', $form->Get());

        $tpl->ParseBlock('AdvancedUserOptions');
        return $tpl->Get();
    }

    /**
     * Returns the user-group management of a selected user
     *
     * @access  public
     * @param   string  $group  Group ID
     * @return  string
     */
    function GetUserGroupUI($group)
    {
        $tpl = new Jaws_Template('gadgets/Users/templates/');
        $tpl->Load('GroupUsers.html');
        $tpl->SetBlock('GroupUsers');
        $tpl->SetVariable('base_script', BASE_SCRIPT);

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $model = new Jaws_User();

        $combo =& Piwi::CreateWidget('CheckButtons', 'group_users', 'vertical');
        $combo->setColumns(1);
        $type = $GLOBALS['app']->Session->IsSuperAdmin() ? 0 : 1;
        $users = $model->GetAllUsers('name', null, $type);
        foreach ($users as $user) {
            $isInGroup = false;
            if ($model->UserIsInGroup($user['id'], $group)) {
                $isInGroup = true;
            }
            $combo->AddOption($user['username'] . ' (' . $user['name']. ')', $user['id'], null, $isInGroup);
        }

        $tpl->SetVariable('group_users_combo', $combo->Get());
        $tpl->SetVariable('title', _t('USERS_GROUPS_MARK_USERS'));
        $tpl->ParseBlock('GroupUsers');
        return $tpl->Get();
    }


    /**
     * Edit properties
     *
     * @access  public
     * @return  string HTML content
     */
    function Properties()
    {
        $this->AjaxMe('script.js');
        $tpl = new Jaws_Template('gadgets/Users/templates/');
        $tpl->Load('Properties.html');
        $tpl->SetBlock('Properties');

        $form =& Piwi::CreateWidget('Form', BASE_SCRIPT, 'post');
        $form->Add(Piwi::CreateWidget('HiddenEntry', 'gadget', 'Users'));
        $form->Add(Piwi::CreateWidget('HiddenEntry', 'action', 'SaveProperties'));

        $priority =& Piwi::CreateWidget('Combo', 'priority');
        $priority->SetTitle(_t('USERS_PROPERTIES_PRIORITY'));
        $priority->AddOption(_t('USERS_PROPERTIES_PRIORITY_UGD'), 'user, groups, default');
        $priority->AddOption(_t('USERS_PROPERTIES_PRIORITY_GUD'), 'groups, user, default');
        $priority->AddOption(_t('USERS_PROPERTIES_PRIORITY_UD'), 'user, default');
        $priority->AddOption(_t('USERS_PROPERTIES_PRIORITY_GD'), 'groups, default');
        $priority->SetDefault($GLOBALS['app']->ACL->GetPriority());

        $authmethod =& Piwi::CreateWidget('Combo', 'auth_method');
        $authmethod->SetTitle(_t('USERS_PROPERTIES_AUTH_METHOD'));
        foreach ($GLOBALS['app']->GetAuthMethods() as $method) {
            $authmethod->AddOption($method, $method);
        }
        $authmethod->SetDefault($GLOBALS['app']->Registry->Get('/config/auth_method'));

        $anonRegister =& Piwi::CreateWidget('Combo', 'anon_register');
        $anonRegister->SetTitle(_t('USERS_PROPERTIES_ANON_REGISTER'));
        $anonRegister->AddOption(_t('GLOBAL_YES'), 'true');
        $anonRegister->AddOption(_t('GLOBAL_NO'), 'false');
        $anonRegister->SetDefault($GLOBALS['app']->Registry->Get('/config/anon_register'));

        $anonType =& Piwi::CreateWidget('Combo', 'anon_type', _t('USERS_PROPERTIES_ANON_TYPE'));
        $anonType->SetID('anon_type');
        $anonType->AddOption(_t('USERS_USERS_TYPE_ADMIN'),  1);
        $anonType->AddOption(_t('USERS_USERS_TYPE_NORMAL'), 2);
        $anonType->SetDefault($GLOBALS['app']->Registry->Get('/config/anon_type'));

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();

        $anonGroup =& Piwi::CreateWidget('Combo', 'anon_group');
        $anonGroup->SetID('anon_group');
        $anonGroup->SetTitle(_t('USERS_PROPERTIES_ANON_GROUP'));
        $anonGroup->AddOption(_t('USERS_GROUPS_NOGROUP'), 0);
        $groups = $userModel->GetAllGroups('title');
        if (!Jaws_Error::IsError($groups)) {
            foreach ($groups as $group) {
                $anonGroup->AddOption($group['title'], $group['id']);
            }
        }
        $anonGroup->SetDefault($GLOBALS['app']->Registry->Get('/config/anon_group'));

        $passRecovery =& Piwi::CreateWidget('Combo', 'password_recovery');
        $passRecovery->SetTitle(_t('USERS_PROPERTIES_PASS_RECOVERY'));
        $passRecovery->AddOption(_t('GLOBAL_YES'), 'true');
        $passRecovery->AddOption(_t('GLOBAL_NO'), 'false');
        $passRecovery->SetDefault($GLOBALS['app']->Registry->Get('/gadgets/Users/password_recovery'));

        include_once JAWS_PATH . 'include/Jaws/Widgets/FieldSet.php';
        $fieldset = new Jaws_Widgets_FieldSet('');
        $fieldset->SetTitle('vertical');

        $fieldset->Add($priority);
        $fieldset->Add($authmethod);
        $fieldset->Add($anonRegister);
        $fieldset->Add($anonType);
        $fieldset->Add($anonGroup);
        $fieldset->Add($passRecovery);

        $form->Add($fieldset);

        $buttons =& Piwi::CreateWidget('HBox');
        $buttons->SetStyle(_t('GLOBAL_LANG_DIRECTION')=='rtl'?'float: left;' : 'float: right;');

        $save =& Piwi::CreateWidget('Button', 'save', _t('GLOBAL_SAVE'), STOCK_SAVE);
        $save->AddEvent(ON_CLICK, 'javascript: saveSettings();');

        $buttons->Add($save);
        $form->Add($buttons);

        $tpl->SetVariable('form', $form->Get());
        $tpl->SetVariable('menubar', $this->MenuBar('Properties'));

        $tpl->ParseBlock('Properties');

        return $tpl->Get();
    }
}
