<?php
/**
 * Users Core Gadget
 *
 * @category   GadgetModel
 * @package    Users
 * @author     Jonathan Hernandez <ion@suavizado.com>
 * @copyright  2004-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
class UsersAdminModel extends Jaws_Model
{
    var $_Name = 'Users';

    /**
     * Installs the gadget
     *
     * @access       public
     * @return       true on successful installation, Jaws_Error otherwise
     */
    function InstallGadget()
    {
        $result = $this->installSchema('schema.xml');
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        //registry keys.
        $GLOBALS['app']->Registry->NewKey('/gadgets/Users/pluggable', 'false');
        $GLOBALS['app']->Registry->NewKey('/gadgets/Users/password_recovery', 'false');
        $GLOBALS['app']->Registry->NewKey('/gadgets/Users/register_notification', 'true');

        //Create the group 'Jaws_Users'
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User;
        $userModel->addGroup('users', 'Users', '', false); //Don't check if it returns true or false
        
        return true;
    }

    /**
     * Update the gadget
     *
     * @access  public
     * @param   string  $old    Current version (in registry)
     * @param   string  $new    New version (in the $gadgetInfo file)
     * @return  boolean  Success/Failure (Jaws_Error)
     */
    function UpdateGadget($old, $new)
    {
        if (version_compare($old, '0.8.0', '<')) {
            $result = $this->installSchema('0.8.0.xml', '', "$old.xml");
            if (Jaws_Error::IsError($result)) {
                return $result;
            }
        }

        $result = $this->installSchema('schema.xml', '', '0.8.0.xml');
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        if (version_compare($old, '0.8.0', '<')) {
            //Create the group 'Jaws_Users'
            require_once JAWS_PATH . 'include/Jaws/User.php';
            $userModel = new Jaws_User;
            $userModel->addGroup('users', 'Users', '', false); //Don't check if it returns true or false
        }

        // Registry keys
        $GLOBALS['app']->Registry->NewKey('/gadgets/Users/register_notification', 'true');

        return true;
    }

    /**
     * Get ACL permissions of a given user
     *
     * @access  public
     * @param   string  $username  Username
     * @return  array   Array with ACL Keys
     */
    function GetUserACLKeys($username)
    {
        $acls = $GLOBALS['app']->ACL->GetAclPermissions($username, false);
        $perms = array();
        if (is_array($acls)) {
            foreach ($acls as $gadget => $keys) {
                $g = $GLOBALS['app']->LoadGadget($gadget, 'Info');
                if (Jaws_Error::IsError($g)) {
                    continue;
                }

                if (!Jaws_Gadget::IsGadgetUpdated($gadget)) {
                    continue;
                }

                if (!isset($perms[$gadget])) {
                    $perms[$gadget] = array();
                }

                foreach ($keys as $k) {
                    $aclkey = '/ACL'.str_replace('/ACL/users/'.$username, '', $k['name']);

                    $perms[$gadget][$aclkey] = array(
                        'desc'  => $g->GetACLDescription($aclkey),
                        'value' => $k['value'],
                        'name'  => $k['name'],
                    );
                }
            }

            ksort($perms);
            return $perms;
        }

        return false;
    }

    /**
     * Returns an array with the ACL keys of a given group
     *
     * @access  public
     * @param   int     $guid   Group's ID
     * @return  array   Array with ACL Keys
     */
    function GetGroupACLKeys($guid)
    {
        $acls = $GLOBALS['app']->ACL->GetGroupAclPermissions($guid);
        $perms = array();
        if (is_array($acls)) {
            foreach ($acls as $gadget => $keys) {
                if (!Jaws_Gadget::IsGadgetUpdated($gadget)) {
                    continue;
                }

                $g = $GLOBALS['app']->LoadGadget($gadget, 'Info');
                if (Jaws_Error::IsError($g)) {
                    continue;
                }

                if (!isset($perms[$gadget])) {
                    $perms[$gadget] = array();
                }

                foreach ($keys as $k) {
                    $aclkey = '/ACL'.str_replace('/ACL/groups/'.$guid, '', $k['name']);
                    $perms[$gadget][$aclkey] = array(
                        'desc'  => $g->GetACLDescription($aclkey),
                        'value' => $k['value'],
                        'name'  => $k['name'],
                    );
                }
            }

            ksort($perms);
            return $perms;
        }

        return false;
    }

    /**
     * Saves only the modified ACL user keys
     *
     * @access  public
     * @param   int     $uid    User' ID
     * @param   array   $keys   ACL Keys
     * @return  boolean Success/Failure(Jaws_Error)
     */
    function UpdateUserACL($uid, $keys)
    {
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();
        $currentUser = $GLOBALS['app']->Session->GetAttribute('user_id');
        if ($user = $userModel->GetInfoById($uid)) {
            //Load user keys
            $GLOBALS['app']->ACL->LoadAllFiles();
            $GLOBALS['app']->ACL->LoadKeysOf($user['username'], 'users');
            foreach($keys as $key => $value) {
                //check user permission for this key
                $expkey = explode('/', $key);
                $aclkey = end($expkey);
                $gadget = prev($expkey);
                if (!$GLOBALS['app']->Session->GetPermission($gadget, $aclkey)) {
                    continue;
                }

                //Get the current value
                if ($key == '/ACL/users/' . $user['username'] . '/gadgets/ControlPanel/default' &&
                    $value === false && $uid == $currentUser)
                {
                    return new Jaws_Error(_t('USERS_USERS_CANT_AUTO_TURN_OFF_CP'), _t('USERS_NAME'));
                }

                $valueString  = $value === true  ? 'true' : 'false';
                $currentValue = $GLOBALS['app']->ACL->Get($key);
                if ($currentValue === null) {
                    $GLOBALS['app']->ACL->NewKey($key, $valueString);
                } else {
                    if (is_bool($currentValue)) {
                        $currentValue = ($currentValue === true) ? 'true' : 'false';
                    }
                    if ($currentValue != $valueString) {
                        $GLOBALS['app']->ACL->Set($key, $valueString);
                    }
                }
            }
            return true;
        }

        return new Jaws_Error(_t('USERS_USER_NOT_EXIST'), _t('USERS_NAME'));
    }

    /**
     * Resets all ACL keys assigned to an user
     *
     * @access  public
     * @param   int     $uid    User' ID
     * @return  boolean Success/Failure(Jaws_Error)
     */
    function ResetUserACL($uid)
    {
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();
        if ($user = $userModel->GetInfoById($uid)) {
            $GLOBALS['app']->ACL->DeleteUserACL($user['username']);
            return true;
        }
        return new Jaws_Error(_t('USERS_USER_NOT_EXIST'), _t('USERS_NAME'));
    }

    /**
     * Saves only the modified ACL group keys
     *
     * @access  public
     * @param   int     $guid   Group ID
     * @param   array   $keys   ACL Keys
     */
    function UpdateGroupACL($guid, $keys)
    {
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();

        if ($group = $userModel->GetGroupInfoById($guid)) {
            $GLOBALS['app']->ACL->LoadAllFiles();
            $GLOBALS['app']->ACL->LoadKeysOf($guid, 'groups');
            foreach ($keys as $key => $value) {
                $valueString  = $value === true  ? 'true' : 'false';
                $currentValue = $GLOBALS['app']->ACL->Get($key);
                if ($currentValue === null) {
                    $GLOBALS['app']->ACL->NewKey($key, $valueString);
                } else {
                    if (is_bool($currentValue)) {
                        $currentValue = ($currentValue === true) ? 'true' : 'false';
                    }

                    if ($currentValue != $valueString) {
                        $GLOBALS['app']->ACL->Set($key, $valueString);
                    }
                }
            }
            return true;
        }
        return new Jaws_Error(_t('USERS_GROUPS_GROUP_NOT_EXIST'), _t('USERS_NAME'));
    }

    /**
     * Resets all ACL keys assigned to a group
     *
     * @access  public
     * @param   int     $guid   Group ID
     * @return  array   Response (notice or error)
     */
    function ResetGroupACL($guid)
    {
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();

        if ($group = $userModel->GetGroupInfoById($guid)) {
            $GLOBALS['app']->ACL->DeleteGroupACL($guid);
            return true;
        }
        return new Jaws_Error(_t('USERS_GROUPS_GROUP_NOT_EXIST'), _t('USERS_NAME'));
    }

    /**
     * Save user config settings
     *
     * @access  public
     * @param   string  $priority  Priority
     * @param   string  $method    Authentication method
     * @param   string  $anon      Anonymous users can auto-register
     * @param   integer $type      User's type
     * @param   integer $group     Default group of anonymous registered user
     * @param   string  $recover   Users can recover their passwords
     * @return  boolean Success/Failure
     */
    function SaveSettings($priority, $method, $anon, $type, $group, $recover)
    {
        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');

        $priority = $xss->parse($priority);
        $method   = $xss->parse($method);
        $anon     = $xss->parse($anon);
        $recover  = $xss->parse($recover);

        $methods = Jaws::getAuthMethods();
        if ($methods !== false && in_array($method, $methods)) {
            $res1 = $GLOBALS['app']->Registry->Set('/config/auth_method', $method);
        } else {
            $res1 = true;
        }
        $res2 = $GLOBALS['app']->ACL->Set('/priority', $priority);
        $res3 = $GLOBALS['app']->Registry->Set('/config/anon_register', $anon);
        $res4 = $GLOBALS['app']->Registry->Set('/config/anon_type',  (int)$type);
        $res5 = $GLOBALS['app']->Registry->Set('/config/anon_group', (int)$group);
        $res6 = $GLOBALS['app']->Registry->Set('/gadgets/Users/password_recovery', $recover);
        if ($res1 === true && $res2 === true && $res3 === true &&
            $res4 === true && $res5 === true && $res6 === true)
        {
            $GLOBALS['app']->Registry->Commit('Users');
            $GLOBALS['app']->Registry->Commit('core');
            $GLOBALS['app']->ACL->Commit('core');
            return true;
        }
        return new Jaws_Error(_t('USERS_PROPERTIES_CANT_UPDATE'), _t('USERS_NAME'));
    }

    /**
     * Add a group of user (by they ids) to a certain group
     *
     * @access  public
     * @param   int     $guid  Group's ID
     * @param   array   $users Array with user id
     * @return  array   Response (notice or error)
     */
    function AddUsersToGroup($guid, $users)
    {
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User();

        $group = $userModel->GetGroupInfoById($guid);
        if (!$group) {
            return new Jaws_Error(_t('USERS_GROUPS_GROUP_NOT_EXIST'), _t('USERS_NAME'));
        }

        $postedUsers = array();
        foreach ($users as $k => $v) {
            $postedUsers[$v] = $v;
        }

        $list = $userModel->GetAllUsers('name', false);
        foreach ($list as $user) {
            if ($userModel->UserIsInGroup($user['id'], $guid)) {
                if (!isset($postedUsers[$user['id']])) {
                    if (!$GLOBALS['app']->Session->IsSuperAdmin() && $user['user_type'] == 0) {
                        continue;
                    }
                    $userModel->DeleteUserFromGroup($user['id'], $guid);
                }
            } else {
                if (isset($postedUsers[$user['id']])) {
                    $userModel->AddUserToGroup($user['id'], $guid);

                }
            }
        }
        return true;
    }
}
