<?php
/**
 * Users Core Gadget
 *
 * @category   GadgetModel
 * @package    Users
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @copyright  2006-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
class UsersModel extends Jaws_Model
{
    var $_Name = 'Users';

    /**
     * Creates a valid(registered) n user for an anonymous user
     *
     * @access  public
     * @param   string  $username  Username
     * @param   string  $email     User's email
     * @param   string  $url       User's url
     * @param   string  $realname  User's realname
     * @param   string  $password  Password
     * @param   string  $p_check   Password check (to verify)
     * @param   boolean $random    Create a random password?
     * @param   string  $user_type User's type
     * @param   string  $group     Default user group
     * @return  boolean Success/Failure
     */
    function CreateUser($username, $email, $url, $realname, $password,
        $p_check, $random, $user_type = 2, $group = null)
    {
        require_once JAWS_PATH . 'include/Jaws/User.php';

        if (trim($username) == '' || trim($realname) == '' || trim($email)  == '') {
            return _t('USERS_USERS_INCOMPLETE_FIELDS');
        }

        $jUser    = new Jaws_User;
        if ($random === true) {
            include_once 'Text/Password.php';
            $password = Text_Password::create(8, 'pronounceable', 'alphanumeric');
        } else {
            if (trim($password) == '') {
                return _t('USERS_USERS_INCOMPLETE_FIELDS');
            }
            if ($password != $p_check) {
                return _t('USERS_USERS_PASSWORDS_DONT_MATCH');
            }
        }

        //Check username
        if (!preg_match('/^[a-z0-9]+$/i', $username) || strlen($username) < 3) {
            return _t('USERS_REGISTER_USERNAME_NOT_VALID');
        }

        require_once JAWS_PATH . 'libraries/pear/Validate.php';
        if (Validate::email($email, true) === false) {
            return _t('USERS_REGISTER_EMAIL_NOT_VALID');
        }

        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $user_type = is_numeric($user_type)? $user_type : 2;
        $result = $jUser->AddUser($username, $realname, $email, $url, md5($password), $user_type);
        if ($result === false) {
            return _t('USERS_USERS_NOT_CREATED', $xss->filter($username));
        }

        if (!is_null($group) && is_numeric($group)) {
            $jUser->AddUserToGroup($result, $group);
        }

        require_once JAWS_PATH . 'include/Jaws/Mail.php';
        $mail = new Jaws_Mail;

        $site_url    = $GLOBALS['app']->getSiteURL() . '/';
        $site_name   = $GLOBALS['app']->Registry->Get('/config/site_name');
        $owner_name  = $GLOBALS['app']->Registry->Get('/config/owner_name');
        $owner_email = $GLOBALS['app']->Registry->Get('/config/owner_email');
        // User wanted a random password? send the password
        if ($random === true) {
            $tpl = new Jaws_Template('gadgets/Users/templates/');
            $tpl->Load('RandomPassword.txt');
            $tpl->SetBlock('RandomPassword');
            $tpl->SetVariable('say_hello', _t('USERS_REGISTER_HELLO', $xss->filter($username)));
            $tpl->SetVariable('registered', _t('USERS_REGISTER_REGISTERED_MAIL_MSG', 
                                               $site_url,
                                               date('Y-m-d')));
            $tpl->SetVariable('ip_name', _t('GLOBAL_IP'));
            $tpl->SetVariable('password_name', _t('USERS_USERS_PASSWORD'));
            $tpl->SetVariable('username_name', _t('USERS_USERS_USERNAME'));
            $tpl->SetVariable('ip_value', $_SERVER['REMOTE_ADDR']);
            $tpl->SetVariable('password_value', $xss->filter($password));
            $tpl->SetVariable('username_value', $xss->filter($username));
            $tpl->SetVariable('thanks', _t('GLOBAL_THANKS'));
            $tpl->SetVariable('site-name', $site_name);
            $tpl->SetVariable('site-url', $site_url);
            $tpl->ParseBlock('RandomPassword');
            $message = $tpl->Get();

            $subject = _t('USERS_REGISTER_SUBJECT', $site_name);
            $mail->SetHeaders($email, $site_name, $owner_email, $subject);
            $mail->AddRecipient($email);
            $mail->SetBody($message, 'text');
            $mresult = $mail->send();
            if (Jaws_Error::IsError($mresult)) {
                $jUser->DeleteUser($result);
                return _t('USERS_REGISTER_NOT_CREATED_SENDMAIL', $xss->filter($email));
            }
        }

        //Send an email to website owner
        $mail->ResetValues();
        if ($GLOBALS['app']->Registry->Get('/gadgets/Users/register_notification') == 'true') {
            $tpl = new Jaws_Template('gadgets/Users/templates/');
            $tpl->Load('NewUser.txt');
            $tpl->SetBlock('NewUser');
            $tpl->SetVariable('say_hello', _t('USERS_REGISTER_HELLO', $owner_name));
            $tpl->SetVariable('registered', _t('USERS_REGISTER_OWNER_MAIL_MSG'));
            $tpl->SetVariable('username_name', _t('USERS_USERS_USERNAME'));
            $tpl->SetVariable('username', $xss->filter($username));
            $tpl->SetVariable('realname_name', _t('USERS_USERS_REALNAME'));
            $tpl->SetVariable('realname', $xss->filter($realname));
            $tpl->SetVariable('email_name', _t('GLOBAL_EMAIL'));
            $tpl->SetVariable('email', $xss->filter($email));
            $tpl->SetVariable('ip_name', _t('GLOBAL_IP'));
            $tpl->SetVariable('ip_value', $_SERVER['REMOTE_ADDR']);
            $tpl->SetVariable('thanks', _t('GLOBAL_THANKS'));
            $tpl->SetVariable('site-name', $site_name);
            $tpl->SetVariable('site-url', $site_url);
            $tpl->ParseBlock('NewUser');
            $message = $tpl->Get();


            $subject = _t('USERS_REGISTER_SUBJECT', $site_name);
            $mail->SetHeaders($owner_email, $site_name, $email, $subject);
            $mail->AddRecipient($owner_email);
            $mail->SetBody($message, 'text');
            $mresult = $mail->send();
            if (Jaws_Error::IsError($mresult)) {
                //do nothing
            }
        }

        return true;
    }

    /**
     * Updates the profile of an user
     *
     * @access  public
     * @param   int      $uid       User's ID
     * @param   string   $username  Username
     * @param   string   $name      User's real name
     * @param   string   $email     User's email
     * @param   string   $url       User's url
     * @param   string   $password  Password
     * @param   boolean  $uppass    Really updte the user password?
     * @return  mixed    True (Success) or Jaws_Error (failure)
     */
    function UpdateAccount($uid, $username, $name, $email, $url, $password)
    {
        if (trim($name) == '' || trim($email) == '') {
            return new Jaws_Error(_t('USERS_USERS_INCOMPLETE_FIELDS'), _t('USERS_NAME'));
        }

        require_once JAWS_PATH . 'libraries/pear/Validate.php';
        if (Validate::email($email, true) === false) {
            return new Jaws_Error(_t('USERS_REGISTER_EMAIL_NOT_VALID'), _t('USERS_NAME'));
        } 

        require_once JAWS_PATH . 'include/Jaws/User.php';
        $jUser    = new Jaws_User;
        $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
        $result = $jUser->UpdateUser($uid, 
                                     $xss->parse($username),
                                     $xss->parse($name),
                                     $xss->parse($email),
                                     $xss->parse($url),
                                     md5($password));
        //TODO: catch error
        return true;
    }

    /**
     * Checks if user/email are valid, if they are generates a recovery secret
     * key and sends it to the user
     *
     * @access  public
     * @param   string  $username  Username
     * @param   string  $email     Email
     * @return  boolean Success/Failure
     */
    function SendRecoveryKey($username, $email)
    {
        require_once JAWS_PATH . 'include/Jaws/User.php';
        
        $userModel = new Jaws_User;
        $info = $userModel->GetInfoByUsername($username);
        if (Jaws_Error::IsError($info)) {
            return $info;
        }

        if (!isset($info['email']) || ($info['email'] != $email)) {
            return new Jaws_Error(_t('USERS_USER_NOT_EXIST'));                
        }

        $secretKey = md5(uniqid(rand(), true)) . time() . floor(microtime()*1000);
        $result    = $userModel->ChangeRecoveryKey($info['id'], $secretKey);
        if ($result === true) {
            $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');

            $site_url    = $GLOBALS['app']->getSiteURL() . '/';
            $site_name   = $GLOBALS['app']->Registry->Get('/config/site_name');
            $owner_name  = $GLOBALS['app']->Registry->Get('/config/owner_name');
            $owner_email = $GLOBALS['app']->Registry->Get('/config/owner_email');

            $tpl = new Jaws_Template('gadgets/Users/templates/');
            $tpl->Load('RecoverPassword.txt');
            $tpl->SetBlock('RecoverPassword');
            $tpl->SetVariable('username', $xss->filter($username));
            $tpl->SetVariable('realname', $xss->filter($info['name']));
            $tpl->SetVariable('message', _t('USERS_FORGOT_MAIL_MESSAGE'));
            $tpl->SetVariable('url', _t('GLOBAL_URL'));
            $tpl->SetVariable('url_value',
                              $GLOBALS['app']->Map->GetURLFor('Users', 'ChangePassword',
                                                              array('key' => $secretKey), true, 'site_url'));
            $tpl->SetVariable('ip', _t('GLOBAL_IP'));
            $tpl->SetVariable('ip_value', $_SERVER['REMOTE_ADDR']);
            $tpl->SetVariable('thanks', _t('GLOBAL_THANKS'));
            $tpl->SetVariable('site-name', $site_name);
            $tpl->SetVariable('site-url', $site_url);
            $tpl->ParseBlock('RecoverPassword');

            $message = $tpl->Get();            
            $subject = _t('USERS_FORGOT_REMEMBER', $site_name);

            require_once JAWS_PATH . 'include/Jaws/Mail.php';
            $mail = new Jaws_Mail;
            $mail->SetHeaders($email, $owner_name, $owner_email, $subject);
            $mail->AddRecipient($email);
            $mail->SetBody($message, 'text');
            $mresult = $mail->send();
            if (Jaws_Error::IsError($mresult)) {
                return new Jaws_Error(_t('USERS_FORGOT_ERROR_SENDING_MAIL'));
            } else {
                return true;
            }
        } else {
            return new Jaws_Error(_t('USERS_FORGOT_ERROR_SENDING_MAIL'));
        }
    }

    /**
     * Changes a password from a given key
     *
     * @access  public
     * @param   string   $key   Recovery key
     * @return  boolean
     */
    function ChangePassword($key)
    {
        require_once JAWS_PATH . 'include/Jaws/User.php';
        
        $jUser = new Jaws_User;
        if ($id = $jUser->GetIDByRecoveryKey($key)) {
            $info = $jUser->GetInfoById($id);

            include_once 'Text/Password.php';
            $password = Text_Password::create(8, 'pronounceable', 'alphanumeric');

            $res1 = $jUser->ChangeRecoveryKey($id);
            $res2 = $jUser->UpdateUser($id, 
                                       $info['username'],
                                       $info['name'],
                                       $info['email'],
                                       $info['url'],
                                       md5($password));

            if ($res1 === false || $res2 === false) {
                return new Jaws_Error(_t('USERS_FORGOT_ERROR_CHANGING_PASSWORD'));
            }
            
            $xss = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');

            $site_url    = $GLOBALS['app']->getSiteURL() . '/';
            $site_name   = $GLOBALS['app']->Registry->Get('/config/site_name');
            $owner_name  = $GLOBALS['app']->Registry->Get('/config/owner_name');
            $owner_email = $GLOBALS['app']->Registry->Get('/config/owner_email');

            $tpl = new Jaws_Template('gadgets/Users/templates/');
            $tpl->Load('NewPassword.txt');
            $tpl->SetBlock('NewPassword');
            $tpl->SetVariable('username', $xss->filter($info['username']));
            $tpl->SetVariable('realname', $xss->filter($info['name']));
            $tpl->SetVariable('password', $password);
            $tpl->SetVariable('message', _t('USERS_FORGOT_PASSWORD_CHANGED_MESSAGE', 
                                            $xss->filter($info['username'])));
            $tpl->SetVariable('password_name', _t('USERS_USERS_PASSWORD'));
            $tpl->SetVariable('username_name', _t('USERS_USERS_USERNAME'));
            $tpl->SetVariable('thanks', _t('GLOBAL_THANKS'));
            $tpl->SetVariable('site-name', $site_name);
            $tpl->SetVariable('site-url', $site_url);
            $tpl->ParseBlock('NewPassword');

            $message = $tpl->Get();            
            $subject = _t('USERS_FORGOT_PASSWORD_CHANGED_SUBJECT');

            require_once JAWS_PATH . 'include/Jaws/Mail.php';
            $mail = new Jaws_Mail;

            $mail->SetHeaders($info['email'], $owner_name, $owner_email, $subject);
            $mail->AddRecipient($info['email']);
            $mail->SetBody($message, 'text');
            $mresult = $mail->send();
            if (Jaws_Error::IsError($mresult)) {
                return new Jaws_Error(_t('USERS_FORGOT_ERROR_SENDING_MAIL'));
            } else {
                return true;
            }
        } else {
            return new Jaws_Error(_t('USERS_FORGOT_KEY_NOT_VALID'));
        }
    }

}