<?php
/**
 * Main application, the core ;-)
 *
 * @category   Application
 * @package    Core
 * @author     Jonathan Hernandez <ion@suavizado.com>
 * @author     Helgi ormar orbjrnsson <dufuz@php.net>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2005-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
class Jaws
{
    /**
     * The application's theme.
     * @var    string
     * @access protected
     */
    var $_Theme = 'jaws';

    /**
     * The language the application is running in.
     * @var    string
     * @access protected
     */
    var $_Language = 'en';

    /**
     * The calendar type.
     * @var    string
     * @access protected
     */
    var $_CalendarType = 'Gregorian';

    /**
     * The calendar language the application is running in.
     * @var    string
     * @access protected
     */
    var $_CalendarLanguage = 'en';

    /**
     * The editor application is using
     * @var    string
     * @access protected
     */
    var $_Editor = null;

    /**
     * The user timezone
     * @var    string
     * @access protected
     */
    var $_UserTimezone = null;

    /**
     * Browser flag
     * @var    string
     * @access protected
     */
    var $_BrowserFlag = '';

    /**
     * Browser HTTP_ACCEPT_ENCODING
     * @var    string
     * @access protected
     */
    var $_BrowserEncoding = '';

    /**
     * Should application use layout?
     * @var    boolean
     * @access protected
     */
    var $_UseLayout = false;

    /**
     * Store gadget object for later use so we aren't running
     * around with multiple copies
     * @var array
     * @access protected
     */
    var $_Gadgets = array();

    /**
     *
     * @var array
     * @access protected
     */
    var $_Classes = array();

    /**
     * Does everything needed to get the application to a usable state.
     *
     * @return void
     * @access public
     */
    function create()
    {
        $this->loadClass('UTF8', 'Jaws_UTF8');
        $this->loadClass('Translate', 'Jaws_Translate');
        $this->loadClass('Registry', 'Jaws_Registry');
        $this->Registry->Init();
        $this->InstanceSession();
  
        $this->loadDefaults();
        $this->Translate->Init($this->_Language);

        // This is needed for all gadgets
        require_once JAWS_PATH . 'include/Jaws/Gadget.php';
        require_once JAWS_PATH . 'include/Jaws/Template.php';

        $this->loadClass('Map', 'Jaws_URLMapping');
        $this->Map->Load();
    }

    /**
     * Set the language and theme, first based on session data, then on application defaults.
     *
     * @return void
     * @access public
     */
    function loadDefaults()
    {
        if (APP_TYPE == 'web') {
            $cookie_precedence = ($this->Registry->Get('/config/cookie_precedence') == 'true');

            $this->_Theme            = $this->Session->GetAttribute('theme');
            $this->_CalendarLanguage = $this->Session->GetAttribute('calendarlanguage');
            $this->_Editor           = $this->Session->GetAttribute('editor');
            $this->_UserTimezone     = $this->Session->GetAttribute('timezone');

            if (is_null($this->_UserTimezone)) {
                if ($cookie_precedence && $this->Session->GetCookie('timezone')) {
                    $this->_UserTimezone = $this->Session->GetCookie('timezone');
                } else {
                    $this->_UserTimezone = $this->Registry->Get('/config/timezone');
                }
            }

            if (empty($this->_Theme)) {
                if ($cookie_precedence && $this->Session->GetCookie('theme')) {
                    $this->_Theme = $this->Session->GetCookie('theme');
                } else {
                    $this->_Theme = $this->Registry->Get('/config/theme');
                }
            }

            if (JAWS_SCRIPT == 'admin') {
                $userLanguage    = $this->Session->GetAttribute('language');
                $this->_Language = empty($userLanguage)? $this->Registry->Get('/config/admin_language') : $userLanguage;
            } elseif (JAWS_SCRIPT == 'index') {
                if ($cookie_precedence && $this->Session->GetCookie('language')) {
                    $this->_Language = $this->Session->GetCookie('language');
                } else {
                    $this->_Language = $this->Registry->Get('/config/site_language');
                }
            } else {
                $this->_Language = 'en';
            }

            if ($cookie_precedence && $this->Session->GetCookie('calendar_type')) {
                $this->_CalendarType = $this->Session->GetCookie('calendar_type');
            } else {
                $this->_CalendarType = $this->Registry->Get('/config/calendar_type');
            }

            if (empty($this->_CalendarLanguage)) {
                if ($cookie_precedence && $this->Session->GetCookie('calendar_language')) {
                    $this->_CalendarLanguage = $this->Session->GetCookie('calendar_language');
                } else {
                    $this->_CalendarLanguage = $this->Registry->Get('/config/calendar_language');
                }
            }
        } else {
            $this->_Theme    = $this->Registry->Get('/config/theme');
            if (JAWS_SCRIPT == 'admin') {
                $this->_Language = $this->Registry->Get('/config/admin_language');
            } elseif (JAWS_SCRIPT == 'index') {
                $this->_Language = $this->Registry->Get('/config/site_language');
            } else {
                $this->_Language = 'en';
            }

            $this->_CalendarType = $this->Registry->Get('/config/calendar_type');
            $this->_CalendarLanguage = $this->Registry->Get('/config/calendar_language');
        }

        if (empty($this->_Editor)) {
            $this->_Editor = $this->Registry->Get('/config/editor');
        }

        require_once 'Net/Detect.php';
        $bFlags = explode(',', $this->Registry->Get('/config/browsers_flag'));
        $this->_BrowserFlag = Net_UserAgent_Detect::getBrowser($bFlags);
    }

    /**
     * Setup the applications session.
     *
     * @return void
     * @access public
     */
    function InstanceSession()
    {
        require_once JAWS_PATH . 'include/Jaws/Session.php';
        $this->Session =& Jaws_Session::factory();
        $this->Session->Init();
    }

    /**
     * Setup the applications cache.
     *
     * @return void
     * @access public
     */
    function InstanceCache()
    {
        require_once JAWS_PATH . 'include/Jaws/Cache.php';
        $this->Cache =& Jaws_Cache::factory();
    }

    /**
     * Setup the applications Layout object.
     *
     * @return void
     * @access public
     */
    function InstanceLayout()
    {
        $this->loadClass('Layout', 'Jaws_Layout');
        $this->Layout->SetTheme($this->_Theme);
        $this->_UseLayout = true;
    }

    /**
     * Get the boolean answer if application is using a layout
     *
     * @return boolean
     * @access public
     */
    function IsUsingLayout()
    {
        return $this->_UseLayout;
    }

    /**
     * Get the name of the Theme
     *
     * @access  public
     * @return  string The name of the theme
     */
    function GetTheme()
    {
        // Check if valid theme name
        if (strpos($this->_Theme, '..') !== false ||
            strpos($this->_Theme, '%') !== false ||
            strpos($this->_Theme, '\\') !== false ||
            strpos($this->_Theme, '/') !== false) {
                return new Jaws_Error(_t('GLOBAL_ERROR_INVALID_NAME', 'GetTheme'), 'Getting theme name');
        }
        return $this->_Theme;
    }

    /**
     * Get the default language
     *
     * @access  public
     * @return  string The default language
     */
    function GetLanguage()
    {
        // Check if valid language name
        if (strpos($this->_Language, '..') !== false ||
            strpos($this->_Language, '%') !== false ||
            strpos($this->_Language, '\\') !== false ||
            strpos($this->_Language, '/') !== false) {
                return new Jaws_Error(_t('GLOBAL_ERROR_INVALID_NAME', 'GetLanguage'), 'Getting language name');
        }
        return $this->_Language;
    }

    /**
     * Get the default editor
     *
     * @access  public
     * @return  string The default language
     */
    function GetEditor()
    {
        return $this->_Editor;
    }

    /**
     * Get Browser flag
     *
     * @access  public
     * @return  string The type of browser
     */
    function GetBrowserFlag()
    {
        return $this->_BrowserFlag;
    }

    /**
     * Overwrites the default values the Application use
     *
     * It overwrites the default values with the input values
     * (which should come in an array)
     *
     *  - Theme:            Array key should be named theme
     *  - Language:         Array key should be named language
     *  - CalendarType:     Array key should be named calendartype
     *  - CalendarLanguage: Array key should be named calendarlanguage
     *  - Editor:           Array key should be named editor
     *
     * In the case of Language and CalendarLanguage, if the new values are
     * different from the default ones (or the values that were already loaded)
     * we load the translation stuff again
     *
     * @access  public
     * @param   array   $defaults  New default values
     */
    function OverwriteDefaults($defaults) 
    {
        if (!is_array($defaults)) {
            return;
        }

        $loadLanguageAgain = false;
        foreach($defaults as $key => $value) {
            $key = strtolower($key);
            if (empty($value)) {
                continue;
            }

            switch($key) {
                case 'theme':
                    $this->_Theme = $value;
                    break;

                case 'language':
                    if ($this->_Language != $value) {
                        $loadLanguageAgain = true;
                        $this->_Language = $value;
                    }
                    break;

                case 'calendartype':
                    $this->_CalendarType = $value;
                    break;

                case 'calendarlanguage':
                    if ($this->_CalendarLanguage != $value) {
                        $loadLanguageAgain = true;
                        $this->_CalendarLanguage = $value;
                    }
                    break;

                case 'editor':
                    $this->_Editor = $value;
                    break;

                case 'timezone':
                    $this->_UserTimezone = $value;
                    break;
            }
        }

        if ($loadLanguageAgain) {
            $this->Translate->Init($this->_Language);
        }
    }
    
    /**
     * Get the default language
     *
     * @access  public
     * @return  string The default language
     */
    function GetCalendarType()
    {
        return $this->_CalendarType;
    }

    /**
     * Get the default language
     *
     * @access  public
     * @return  string The default language
     */
    function GetCalendarLanguage()
    {
        return $this->_CalendarLanguage;
    }

    /**
     * Get the available authentication methods
     *
     * @access  public
     * @return  array  Array with available authentication methods
     */
    function GetAuthMethods()
    {
        $path = JAWS_PATH . 'include/Jaws/AuthScripts';
        if (is_dir($path)) {
            $methods = array();
            $dir = scandir($path);
            foreach ($dir as $method) {
                if (stristr($method, '.php')) {
                    $method = str_replace('.php', '', $method);
                    $methods[$method] = $method;
                }
            }

            return $methods;
        }

        return false;
    }

    /**
     * Loads the gadget file in question, makes a instance and
     * stores it globally for later use so we do not have duplicates
     * of the same instance around in our code.
     *
     * @param string Name of the gadget
     * @param string The file being loaded
     *
     * @access public
     */
    function LoadGadget($gadget, $filename = 'HTML')
    {
        $gadget   = trim($gadget);
        $filename = trim($filename);
        $gadgetname = $gadget . ucfirst($filename);
        $load_registry = true;
        if (!isset($this->_Gadgets[$gadget][$filename])) {
            switch ($filename) {
                case 'Info':
                    $load_registry = false;
                    if (!Jaws::classExists('Jaws_GadgetInfo')) {
                        require_once JAWS_PATH . 'include/Jaws/GadgetInfo.php';
                    }
                    break;
                case 'HTML':
                case 'AdminHTML':
                    if (!Jaws::classExists('Jaws_GadgetHTML')) {
                        require_once JAWS_PATH . 'include/Jaws/GadgetHTML.php';
                    }
                    break;
            }

            $file = JAWS_PATH . 'gadgets/' . $gadget . '/' . $filename . '.php';
            if (file_exists($file)) {
                include_once $file;
            }

            if (!Jaws::classExists($gadgetname)) {
                // return a error
                $error = new Jaws_Error(_t('GLOBAL_ERROR_CLASS_DOES_NOT_EXIST', $gadgetname), 'Gadget class check');
                return $error;
            }

            $obj = new $gadgetname();
            if (Jaws_Error::IsError($obj)) {
                $error = new Jaws_Error(_t('GLOBAL_ERROR_FAILED_CREATING_INSTANCE', $file, $gadgetname), 'Gadget file loading');
                return $error;
            }

            if ($load_registry && (!isset($this->_Gadgets[$gadget]) || !isset($this->_Gadgets[$gadget]['Registry']))) {
                $this->_Gadgets[$gadget]['Registry'] = true;
                if (isset($this->ACL)) {
                    $this->ACL->LoadFile($gadget);
                }
                $this->Registry->LoadFile($gadget);
            }

            $this->_Gadgets[$gadget][$filename] = $obj;
            if (isset($GLOBALS['log'])) {
                $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'Loaded gadget: ' . $gadget . ', File: ' . $filename);
            }
        }

        return $this->_Gadgets[$gadget][$filename];
    }

    /**
     * Set true|false if a gadget has been updated so we don't check it again and again
     *
     * @access  public
     * @param   string  $gadget  Gadget's name
     * @param   string  $status  True if gadget is updated (installed and latest version)
     * @return  void
     */
    function SetGadgetAsUpdated($gadget, $status = true)
    {
        if (!empty($gadget) && !isset($this->_Gadgets[$gadget]['is_updated'])) {
            $this->_Gadgets[$gadget]['is_updated'] = $status;
        }
    }

    /**
     * Returns true or false is gadget has been marked as updated. If the gadget hasn't been marked
     * it returns null.
     *
     * @access  public
     * @param   string  $gadget  Gadget's name
     * @return  void
     */
    function IsGadgetMarkedAsUpdated($gadget)
    {
        if (!empty($gadget) && isset($this->_Gadgets[$gadget]['is_updated'])) {
            return $this->_Gadgets[$gadget]['is_updated'];
        }

        return null;
    }

    /**
     * Gets a list of installed gadgets (using Singleton), it uses
     * the /gadget/enabled_items
     *
     * @access   public
     * @return   array   Array of enabled_items (and updated)
     */
    function GetInstalledGadgets()
    {
        static $installedGadgets;

        if (isset($installedGadgets)) {
            return $installedGadgets;
        }
        $installedGadgets = array();

        $gs = explode(',', $GLOBALS['app']->Registry->Get('/gadgets/enabled_items'));
        $ci = explode(',', $GLOBALS['app']->Registry->Get('/gadgets/core_items'));
        $ci = str_replace(' ', '', $ci);
        $gs = array_merge($gs, $ci);

        if (count($gs) > 0) {
            foreach ($gs as $gadget) {
                if (file_exists(JAWS_PATH . 'gadgets/' . $gadget . '/Info.php')) {
                    if (Jaws_Gadget::IsGadgetUpdated($gadget)) {
                        $installedGadgets[$gadget] = $gadget;
                    }
                }
            }
        }

        return $installedGadgets;
    }

    /**
     * Loads the action file of a gadget
     *
     * @access  public
     * @param   string  $gadget  Gadget's name
     */
    function LoadGadgetActions($gadget)
    {
        if (!isset($this->_Gadgets[$gadget]['actions'])) {
            $file = JAWS_PATH . 'gadgets/' . $gadget . '/Actions.php';

            if (file_exists($file)) {
                $this->Translate->LoadTranslation($gadget, JAWS_PATH . 'gadgets/');
                require_once $file;
                if (isset($actions)) {
                    $tmp = array();

                    // key: Action Name  value: Action Properties
                    foreach ($actions as $aName => $aProps) {
                        if (isset($aProps[2])) {
                            $name = isset($aProps[1]) ? $aProps[1] : '';
                        } else {
                            $name = $aName;
                        }

                        if (!isset($aProps[0])) {
                            $aProps[0] = 'NormalAction';
                        }
                        foreach (explode(",", $aProps[0]) as $type) {
                            $tmp[trim($type)][$aName] = array(
                                                            'name' => $name,
                                                            'mode' => trim($type),
                                                            'desc' => (isset($aProps[2])) ? $aProps[2] : ''
                                                        );
                        }
                    }
                    $this->_Gadgets[$gadget]['actions'] = $tmp;
                } else {
                    $this->_Gadgets[$gadget]['actions'] = array();
                }
            } else {
                $this->_Gadgets[$gadget]['actions'] = array();
            }
        }
    }

    /**
     * Gets the actions of a gadget
     *
     * @access  public
     * @param   string  $gadget  Gadget's name
     * @return  array   Gadget actions
     */
    function GetGadgetActions($gadget)
    {
        if (!isset($this->_Gadgets[$gadget]['actions'])) {
            $this->LoadGadgetActions($gadget);
        }
        return $this->_Gadgets[$gadget]['actions'];
    }

    /**
     * Prepares the jaws Editor
     *
     * @access  public
     * @param   string  $gadget  Gadget that uses the editor (usable for plugins)
     * @param   string  $name    Name of the editor
     * @param   string  $value   Value of the editor/content (optional)
     * @param   string  $label   Label that the editor will have (optional)
     * @return  object  The editor in /config/editor
     */
    function &LoadEditor($gadget, $name, $value = '', $filter = true, $label = '')
    {
        if ($filter && !empty($value)) {
            $xss   = $this->loadClass('XSS', 'Jaws_XSS');
            $value = $xss->filter($value);
        }

        $editor = $this->_Editor;
        $file   = JAWS_PATH . 'include/Jaws/Widgets/' . $editor . '.php';
        if (!file_exists($file)) {
            $editor = 'TextArea';
            $file   = JAWS_PATH . 'include/Jaws/Widgets/' . $editor . '.php';
        }
        $editorClass = "Jaws_Widgets_$editor";

        require_once $file;
        $editor = new $editorClass($gadget, $name, $value, $label);

        return $editor;
    }

    /**
     * Loads the Jaws Date class.
     * Singleton approach.
     *
     * @access public
     */
    function loadDate()
    {
        static $instances;
        if (!isset($instances)) {
            $instances = array();
        }

        $signature = serialize(array('date'));
        if (!isset($instances[$signature])) {
            include_once JAWS_PATH . 'include/Jaws/Date.php';
            $calendar = $this->GetCalendarType();
            $instances[$signature] =& Jaws_Date::factory($calendar);

            if (isset($GLOBALS['log'])) {
                $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'Date class is loaded');
            }
        }

        return $instances[$signature];

//         return $this->loadClass('Date', 'Jaws_Date');
    }

    /**
     * Loads a class from within the Jaws dir
     *
     * @access public
     */
    function loadClass($property, $class)
    {
        if (!isset($this->{$property})) {
            $file = JAWS_PATH . 'include'. DIRECTORY_SEPARATOR . str_replace('_', DIRECTORY_SEPARATOR, $class).'.php';
            if (!file_exists($file)) {
                $error = new Jaws_Error(_t('GLOBAL_ERROR_FILE_DOES_NOT_EXIST', $file), 'File exists check');
                return $error;
            }

            include_once $file;

            if (!$this->classExists($class)) {
                $error = new Jaws_Error(_t('GLOBAL_ERROR_CLASS_DOES_NOT_EXIST', $class), 'Class exists check');
                return $error;
            }

            $this->{$property} = new $class();
            if (Jaws_Error::IsError($this->{$property})) {
                $error = new Jaws_Error(_t('GLOBAL_ERROR_FAILED_CREATING_INSTANCE', $file, $class), 'Class file loading');
                return $error;
            }

            if (isset($GLOBALS['log'])) {
                $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'Loaded class: ' . $class . ', File: ' . $file);
            }
        }

        return $this->{$property};
    }

    /**
     * Stub function for now, it will handle loading files with only include
     * kinda internal include_once
     */
    function loadFile($path)
    {

    }

    /**
     * Verify if an image exists, if not returns a default image (unknown.png)
     *
     * @param   string Image path
     * @return  string The original path if it exists or an unknow.png path
     * @access  public
     */
    function CheckImage($path)
    {
        $path = JAWS_DPATH . $path;
        if (is_file($path)) {
            return $path;
        }

        return JAWS_DPATH . 'images/unknown.png';
    }

    /**
     * Returns the current location (without BASE_SCRIPT)
     *
     * @access  public
     * @return  string   Current location
     */
    function GetURILocation()
    {
        static $location;

        if (isset($location)) {
            return $location;
        }

        $xss = $this->loadClass('XSS', 'Jaws_XSS');
        //TODO: Need to check which SERVER var is allways sent to the server
        if (!isset($_SERVER['REQUEST_URI']) || empty($_SERVER['REQUEST_URI'])) {
            $location = $xss->filter($_SERVER['SCRIPT_NAME']);
        } else {
            $location = $xss->filter($_SERVER['REQUEST_URI']);
        }
        $location = substr($location, 0, stripos($location, BASE_SCRIPT));
        return $location;
    }

    /**
     * Returns the URL of the site
     *
     * @param   string jaws_iurl url of jaws instance
     * @param   string rel_url relative url
     * @access  public
     * @return  string  Site's URL
     */
    function getSiteURL($jaws_iurl = true, $rel_url = false)
    {
        static $site_url;
        if (!isset($site_url)) {
            if (!defined('JAWS_DPATH')) {
                define('JAWS_DPATH', '');
            }

            $cfg_url = isset($GLOBALS['app']->Registry)? $GLOBALS['app']->Registry->Get('/config/site_url') : '';
            if (!empty($cfg_url)) {
                $cfg_url = parse_url($cfg_url);
                if (isset($cfg_url['scheme']) && isset($cfg_url['host'])) {
                    $cfg_url['path'] = isset($cfg_url['path'])? $cfg_url['path'] : '';
                    $site_url = $cfg_url;
                }
            }

            if (!isset($site_url)) {
                $site_url = array();
                $site_url['scheme'] = (isset($_SERVER['HTTPS']) && strtolower($_SERVER['HTTPS']) == 'on')? 'https' : 'http';
                $host = $_SERVER['HTTP_HOST'];
                if (false === strpos($host, ':')) {
                    $site_url['host'] = $host;
                } else {
                    $site_url['host'] = substr($host, 0, strpos($host, ':'));
                    $site_url['port'] = substr($host, strpos($host, ':') + 1);
                }

                $path = strip_tags($_SERVER['PHP_SELF']);
                if (false === stripos($path, BASE_SCRIPT)) {
                    $path = strip_tags($_SERVER['SCRIPT_NAME']);
                    if (false === stripos($path, BASE_SCRIPT)) {
                        $pInfo = isset($_SERVER['PATH_INFO'])? $_SERVER['PATH_INFO'] : '';
                        $pInfo = (empty($pInfo) && isset($_SERVER['ORIG_PATH_INFO']))? $_SERVER['ORIG_PATH_INFO'] : '';
                        $pInfo = (empty($pInfo) && isset($_ENV['PATH_INFO']))? $_ENV['PATH_INFO'] : '';
                        $pInfo = (empty($pInfo) && isset($_ENV['ORIG_PATH_INFO']))? $_ENV['ORIG_PATH_INFO'] : '';
                        $pInfo = strip_tags($pInfo);
                        if (!empty($pInfo)) {
                            $path = substr($path, 0, strpos($path, $pInfo)+1);
                        }
                    }
                }
                $site_url['path'] = substr($path, 0, stripos($path, BASE_SCRIPT)-1);
            }
        }

        $url = $site_url['path'];
        if (!$jaws_iurl && JAWS_DPATH!='') {
            $url = Jaws_Utils::clean_path($url .'/'. JAWS_DPATH);
        }

        if (!$rel_url) {
            $url = $site_url['scheme'] . '://' . $site_url['host'] . (isset($site_url['port'])? ':'.$site_url['port'] : '') . $url;
        }

        if (substr($url, -1) == '/') {
            $url = substr($url, 0, -1);
        }

        return $url;
    }

    /**
     * Returns the URL of the data
     *
     * @param   string rel_url relative url
     * @access  public
     * @return  string  Data's URL
     */
    function getDataURL($full_url = false)
    {
        $url = substr(JAWS_DATA, strlen(JAWS_IPATH));

        if (DIRECTORY_SEPARATOR !='/') {
            $url = str_replace('\\', '/', $url);
        }
        if ($full_url) {
            $url = $this->getSiteURL() . '/' . $url;
        }
        return $url;
    }

    /**
     * Executes the autoload gadgets
     *
     * @param   public
     */
    function RunAutoload()
    {
        $data    = $GLOBALS['app']->Registry->Get('/gadgets/autoload_items');
        $gadgets = explode(',', $data);
        foreach($gadgets as $gadgetName) {
            $gadget = $this->loadGadget($gadgetName, 'Autoload');
            if (!Jaws_Error::isError($gadget)) {
                if (method_exists($gadget, 'Execute')) {
                    $gadget->Execute();
                }
            }
        }
    }

    /**
     * Returns a gadget hook of a specific gadget
     *
     * @access  public
     * @param   string  $gadget  Gadget we want to load (where the hook is)
     * @param   string  $hook    Gadget hook (the hook name)
     * @return  object  Gadget's hook if it exists or false
     */
    function loadHook($gadget, $hook)
    {
        $hookName = $gadget.$hook.'Hook';
        if (!isset($this->_Classes[$hookName])) {
            $hookFile = JAWS_PATH . 'gadgets/' . $gadget . '/hooks/' . $hook . '.php';
            if (file_exists($hookFile)) {
                include_once $hookFile;
            }

            if (!Jaws::classExists($hookName)) {
                return false;
            }

            $obj = new $hookName();
            $this->_Classes[$hookName] = $obj;
            if (isset($GLOBALS['log'])) {
                $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'Loaded hook: ' . $hook . ' of gadget '. $gadget. ', File: ' . $hookFile);
            }
        }
        return $this->_Classes[$hookName];
    }

    /**
     * Checks if a class exists without triggering __autoload
     *
     * @param   string  classname
     * @return  bool    true success and false on error
     *
     * @access  public
     */
    function classExists($classname)
    {
        if (version_compare(PHP_VERSION, '5.0', '>=')) {
            return class_exists($classname, false);
        }
        return class_exists($classname);
    }

    /**
     * Get Browser accept encoding
     *
     * @access  public
     * @return  string The type of browser
     */
    function GetBrowserEncoding()
    {
        return $this->_BrowserEncoding;
    }

    /**
     * use native gzip compression?
     *
     * @access  private
     * @return  boolean True or False
     */
    function GZipEnabled()
    {
        static $_GZipEnabled;
        if (!isset($_GZipEnabled)) {
            $this->_BrowserEncoding = (isset($_SERVER['HTTP_ACCEPT_ENCODING']) ? $_SERVER['HTTP_ACCEPT_ENCODING'] : '');
            $this->_BrowserEncoding = strtolower($this->_BrowserEncoding);
            $_GZipEnabled = true;
            if (($this->Registry->Get('/config/gzip_compression') != 'true') ||
                !extension_loaded('zlib') ||
                ini_get('zlib.output_compression') ||
                (ini_get('zlib.output_compression_level') > 0) ||
                (ini_get('output_handler') == 'ob_gzhandler') ||
                (ini_get('output_handler') == 'mb_output_handler') ||
                (strpos($this->_BrowserEncoding, 'gzip') === false))
            {
                $_GZipEnabled = false;
            }
        }

        return $_GZipEnabled;
    }

    /**
     * Is actual agent a robot?
     *
     * @access  private
     * @return  boolean True or False
     */
    function IsAgentRobot()
    {
        static $_IsRobot;
        if (!isset($_IsRobot)) {
            $_IsRobot = false;
            $robots = explode(',', $this->Registry->Get('/config/robots'));
            $robots = array_map('strtolower', $robots);
            $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
            $uagent = strtolower($GLOBALS['app']->XSS->parse($_SERVER['HTTP_USER_AGENT']));
            $ipaddr = $_SERVER['REMOTE_ADDR'];
            foreach($robots as $robot) {
                if (!empty($robot) && (($ipaddr == $robot) || (strpos($uagent, $robot) !== false))) {
                    $_IsRobot = true;
                    break;
                }
            }
        }

        return $_IsRobot;
    }

    /**
     * Get user time
     *
     * @access  private
     * @param   mixed   $time   timestamp
     * @param   string  $format date format
     * @return  boolean True or False
     */
    function UTC2UserTime($time = '', $format = '')
    {
        $time = empty($time)? time() : $time;
        $time = is_int($time)? $time : strtotime($time);
        $time = $time + ($this->_UserTimezone * 3600);
        return empty($format)? $time : date($format, $time);
    }

    /**
     * Get UTC time
     *
     * @access  private
     * @param   mixed   $time   timestamp
     * @param   string  $format date format
     * @return  boolean True or False
     */
    function UserTime2UTC($time, $format = '')
    {
        $time = is_int($time)? $time : strtotime($time);
        $time = $time - ($this->_UserTimezone * 3600);
        return empty($format)? $time : date($format, $time);
    }

}
