<?php
/**
 * Class to manage Jaws Layout
 *
 * @category   Layout
 * @package    Core
 * @author     Jonathan Hernandez <ion@suavizado.com>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2005-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
class Jaws_Layout
{
    /**
     * Model that will be used to get data
     *
     * @var    LayoutJaws_Model
     * @access private
     */
    var $_Model;

    /**
     * Template that will be used to print the data
     *
     * @var    Jaws_Template
     * @access private
     */
    var $_Template;

    /**
     * Name of the theme
     *
     * @var    string
     * @access private
     */
    var $_Theme = '';

    /**
     * Array that will have the meta tags
     *
     * @var    array
     * @access private
     */
    var $_HeadMeta = array();

    /**
     * Array that will have the links meta tags
     *
     * @var    array
     * @access private
     */
    var $_HeadLink = array();

    /**
     * Array that will have the JS links
     *
     * @var    array
     * @access private
     */
    var $_ScriptLink = array();

    /**
     * Array that will contain other info/text
     * that has to go into the <head> part
     *
     * @var    array
     * @access private
     */
    var $_HeadOther = array();

    /**
     * Requested gadget
     *
     * @access private
     * @var    string
     */
    var $_RequestedGadget;

    /**
     * Requested gadget's action
     *
     * @access private
     * @var    string
     */
    var $_RequestedAction;

    /**
     * Current section
     *
     * @access private
     * @var string
     */
    var $_Section = '';

    /**
     * Current section
     *
     * @access private
     * @var string
     */
    var $_SectionAttributes = array();

    /**
     * Returns the current URI location (without BASE_SCRIPT's value)
     *
     * @access  private
     * @var     string
     */
    var $_CurrentLocation;

    /**
     * Page title
     *
     * @access  private
     * @var     string
     */
    var $_Title = null;

    /**
     * Page keywords
     *
     * @access  private
     * @var     array
     */
    var $_Keywords = array();

    /**
     * Page languages
     *
     * @access  private
     * @var     array
     */
    var $_Languages = array();

    /**
     * Initializes the Layout
     *
     * @access  public
     */
    function Jaws_Layout()
    {
        // Set Headers
        header('Content-Type: text/html; charset=utf-8'); //magic, big fix
        header('Cache-Control: no-cache, must-revalidate');
        header('Pragma: no-cache');

        //load default site keywords
        $keywords = $GLOBALS['app']->Registry->Get('/config/site_keywords');
        $this->_Keywords = array_map(array('Jaws_UTF8','trim'), explode(',', $keywords));

        // set default site language
        $this->_Languages[] = $GLOBALS['app']->GetLanguage();

        $this->_Model = $GLOBALS['app']->loadGadget('Layout', 'Model');
        if (Jaws_Error::isError($this->_Model)) {
            Jaws_Error::Fatal("Can't load layout model", __FILE__, __LINE__);
        }
    }

    /**
     * Gets the theme name
     *
     * @access  public
     * @return  string  Theme's name
     */
    function GetTheme()
    {
        return $this->_Theme;
    }

    /**
     * Sets the theme name
     *
     * @access  public
     * @param   string  $theme  Theme's name
     */
    function SetTheme($theme)
    {
        $this->_Theme = $theme;
    }

    /**
     * Gets the current section
     *
     * @access public
     * @return string Current section
     */
    function GetSectionName()
    {
        return $this->_Section;
    }

    /**
     * Is current section wide?
     *
     * @access public
     * @return boolean
     */
    function IsSectionWide()
    {
        return !isset($this->_SectionAttributes['narrow']);
    }

    /**
     */
    function AddJSPrototypeHeadLink()
    {
        // Prototype, Scriptaculous and Response
        $this->AddScriptLink('libraries/prototype/prototype.js');
        $this->AddScriptLink('libraries/prototype/scriptaculous.js');
        $this->AddScriptLink('include/Jaws/Ajax/Response.js');
    }

    /**
     * Loads the template
     *
     * @access  public
     */
    function Load($ajaxEnabled = false)
    {
        if ($GLOBALS['app']->Registry->Get('/config/site_status') == 'disabled' && !$GLOBALS['app']->Session->IsAdmin())
        {
            require_once JAWS_PATH . 'include/Jaws/HTTPError.php';
            echo Jaws_HTTPError::Get(503);
            exit;
        }

        if ($ajaxEnabled || $GLOBALS['app']->Registry->Get('/config/frontend_ajaxed') == 'true') {
            $this->AddJSPrototypeHeadLink();
        }

        if (isset($GLOBALS['app'])) {
            $themeName = $GLOBALS['app']->GetTheme();
            $themeFile = JAWS_DATA . 'themes/' . $themeName . '/' . 'layout.html';
            if (!file_exists($themeFile)) {
                Jaws_Error::Fatal('The layout.html file in your theme ('. $themeName . ') directory doesn\'t exists',
                                 __FILE__,
                                 __LINE__);
            }
        }

        //Check if theme/layout.html file exists
        $this->_Template = new Jaws_Template('', $this->_Theme);
        $this->_Template->Load('layout.html');
        $this->_Template->SetBlock('layout');

        $direction = _t('GLOBAL_LANG_DIRECTION');
        $dir  = $direction == 'rtl' ? '.' . $direction : '';
        $brow = $GLOBALS['app']->GetBrowserFlag();
        $brow = empty($brow)? '' : '.'.$brow;
        $base_url = $GLOBALS['app']->GetSiteURL().'/';
        $site_url = $GLOBALS['app']->Registry->Get('/config/site_url');

        $this->_Template->SetVariable('BASE_URL', $base_url);
        $this->_Template->SetVariable('.dir', $dir);
        $this->_Template->SetVariable('.browser', $brow);
        $this->_Template->SetVariable('site-url', empty($site_url)? $base_url : $site_url);
        $this->_Template->SetVariable('site-name',        $GLOBALS['app']->Registry->Get('/config/site_name'));
        $this->_Template->SetVariable('site-description', $GLOBALS['app']->Registry->Get('/config/site_description'));
        $this->_Template->SetVariable('site-author',      $GLOBALS['app']->Registry->Get('/config/owner_name'));
        $this->_Template->SetVariable('site-copyright',   $GLOBALS['app']->Registry->Get('/config/copyright'));
        
        // Deprecated since 0.8: This is for backwards compatibility
        switch ($GLOBALS['app']->Registry->Get('/config/layoutmode')) {
            case 1: $layoutmode = 'twobar';
                    break;
            case 2: $layoutmode = 'leftbar';
                    break;
            case 3: $layoutmode = 'rightbar';
                    break;
            case 4: $layoutmode = 'nobar';
                    break;
        } 
        $this->_Template->SetVariable('layout-mode-name', $layoutmode);
        
        $this->_Template->SetVariable('encoding', 'utf-8');
    }

    /**
     * Loads the template for head of control panel
     *
     * @access  public
     */
    function LoadControlPanelHead()
    {
        $this->AddHeadLink('gadgets/ControlPanel/resources/public.css', 'stylesheet', 'text/css');
        $this->AddHeadLink(PIWI_URL . 'piwidata/css/default.css', 'stylesheet', 'text/css');
        $this->AddJSPrototypeHeadLink();

        $GLOBALS['app']->LoadGadget('ControlPanel', 'AdminHTML');

        $this->_Template = new Jaws_Template('gadgets/ControlPanel/templates/');
        $this->_Template->Load('Layout.html');
        $this->_Template->SetBlock('layout');

        $base_url = $GLOBALS['app']->GetSiteURL().'/';
        $this->_Template->SetVariable('BASE_URL', $base_url);
        $this->_Template->SetVariable('admin_script', BASE_SCRIPT);
        $this->_Template->SetVariable('site-name',        $GLOBALS['app']->Registry->Get('/config/site_name'));
        $this->_Template->SetVariable('site-description', $GLOBALS['app']->Registry->Get('/config/site_description'));
        $this->_Template->SetVariable('site-author',      $GLOBALS['app']->Registry->Get('/config/owner_name'));
        $this->_Template->SetVariable('site-copyright',   $GLOBALS['app']->Registry->Get('/config/copyright'));
        $this->_Template->SetVariable('control-panel', _t('CONTROLPANEL_NAME'));
        $this->_Template->SetVariable('loading-message', _t('CONTROLPANEL_LOADING'));
        $this->_Template->SetVariable('navigate-away-message', _t('CONTROLPANEL_UNSAVED_CHANGES'));
        $this->_Template->SetVariable('encoding', 'utf-8');
    }

    /**
     * Loads the template for controlpanel
     *
     * @param   string  $gadget Gadget name
     * @access  public
     */
    function LoadControlPanel($gadget)
    {
        $this->_Template->SetBlock('layout/login-info', false);
        $this->_Template->SetVariable('logged-in-as', _t('CONTROLPANEL_LOGGED_IN_AS'));
        $this->_Template->SetVariable('username', $GLOBALS['app']->Session->GetAttribute('username'));
        require_once JAWS_PATH . 'include/Jaws/User.php';
        $userModel = new Jaws_User;
        $userInfo = $userModel->GetInfoByUsername($GLOBALS['app']->Session->GetAttribute('username'));
        $this->_Template->SetVariable('name', $userInfo['name']);
        $this->_Template->SetVariable('email', $userInfo['email']);
        $this->_Template->SetVariable('user_image', $userModel->GetAvatar($userInfo['username'], $userInfo['email']));
        $this->_Template->SetVariable('site-url', $GLOBALS['app']->GetSiteURL());
        $this->_Template->SetVariable('my-account', _t('GLOBAL_MY_ACCOUNT'));
        $this->_Template->SetVariable('my-account-url', BASE_SCRIPT . '?gadget=Users&amp;action=MyAccount');
        $this->_Template->SetVariable('logout', _t('GLOBAL_LOGOUT'));
        $this->_Template->SetVariable('logout-url', BASE_SCRIPT . '?gadget=ControlPanel&amp;action=Logout');
        $this->_Template->ParseBlock('layout/login-info');

        // Set the header thingie for each gadget and the response box
        if (isset($gadget) && ($gadget != 'ControlPanel')){
            $gInfo  = $GLOBALS['app']->loadGadget($gadget, 'Info');
            $docurl = null;
            if (!Jaws_Error::isError($gInfo)) {
                $docurl = $gInfo->GetDoc();
            }
            $gname = _t(strtoupper($gadget) . '_NAME');
            $this->_Template->SetBlock('layout/cptitle');
            $this->_Template->SetVariable('admin_script', BASE_SCRIPT);
            $this->_Template->SetVariable('title-cp', _t('CONTROLPANEL_NAME'));
            $this->_Template->SetVariable('title-name', $gname);
            $this->_Template->SetVariable('icon-gadget', JAWS_DPATH . 'gadgets/'.$gadget.'/images/logo.png');
            $this->_Template->SetVariable('title-gadget', $gadget);
            if (!empty($docurl) && !is_null($docurl)) {
                $this->_Template->SetBlock('layout/cptitle/documentation');
                $this->_Template->SetVariable('src', JAWS_DPATH . 'images/stock/help-browser.png');
                $this->_Template->SetVariable('alt', _t('GLOBAL_READ_DOCUMENTATION'));
                $this->_Template->SetVariable('url', $docurl);
                $this->_Template->ParseBlock('layout/cptitle/documentation');
            }

            if (_t(strtoupper($gadget).'_ADMIN_MESSAGE') != strtoupper($gadget).'_ADMIN_MESSAGE*') {
                $this->_Template->SetBlock('layout/cptitle/description');
                $this->_Template->SetVariable('title-desc', _t(strtoupper($gadget) . '_ADMIN_MESSAGE'));
                $this->_Template->ParseBlock('layout/cptitle/description');
            }
            $this->_Template->ParseBlock('layout/cptitle');
        }

        if ($GLOBALS['app']->Registry->Get('/config/site_status') == 'disabled') {
            $this->_Template->SetBlock('layout/warning');
            $this->_Template->SetVariable('warning', _t('CONTROLPANEL_OFFLINE_WARNING'));
            $this->_Template->ParseBlock('layout/warning');
        }

        $responses = $GLOBALS['app']->Session->PopLastResponse();
        if ($responses) {
            foreach ($responses as $msg_id => $response) {
                $this->_Template->SetBlock('layout/msgbox');
                $this->_Template->SetVariable('msg-css', $response['css']);
                $this->_Template->SetVariable('msg-txt', $response['message']);
                $this->_Template->SetVariable('msg-id', $msg_id);
                $this->_Template->ParseBlock('layout/msgbox');
            }
        }
    }

    /**
     * Changes the site-title with something else
     *
     * @access  public
     * @param   string  $title  New title
     */
    function SetTitle($title)
    {
        $this->_Title = strip_tags($title);
    }

    /**
     * Assign the right head's title
     *
     * @access  public
     */
    function PutTitle()
    {
        if (!empty($this->_Title)) {
            $pageTitle = array($this->_Title, $GLOBALS['app']->Registry->Get('/config/site_name'));
        } else {
            $description = $GLOBALS['app']->Registry->Get('/config/site_description');
            $pageTitle   = array();
            $pageTitle[] = $GLOBALS['app']->Registry->Get('/config/site_name');
            if (!empty($description)) {
                $pageTitle[] = $description;
            }
        }
        $pageTitle = implode(' ' . $GLOBALS['app']->Registry->Get('/config/title_separator').' ', $pageTitle);
        $this->_Template->ResetVariable('site-title', $pageTitle, 'layout');
    }

    /**
     * Add keywords to meta keywords tag
     *
     * @access  public
     * @param   string  $keywords  page keywords
     */
    function AddToMetaKeywords($keywords)
    {
        if (!empty($keywords)) {
            $keywords = array_map(array('Jaws_UTF8','trim'), explode(',', $keywords));
            $this->_Keywords = array_merge($this->_Keywords, $keywords);
        }
    }

    /**
     * Assign the site keywords
     *
     * @access  public
     */
    function PutMetaKeywords()
    {
        $this->_Template->ResetVariable('site-keywords',
                                        strip_tags(implode(', ', $this->_Keywords)),
                                        'layout');
    }

    /**
     * Add a language to meta language tag
     *
     * @access  public
     * @param   string  $language  Language
     */
    function AddToMetaLanguages($language)
    {
        if (!empty($language)) {
            if (!in_array($language, $this->_Languages)) {
                $this->_Languages[] = $language;
            }
        }
    }

    /**
     * Assign the site languages
     *
     * @access  public
     */
    function PutMetaLanguages()
    {
        $this->_Template->ResetVariable('site-languages',
                                        strip_tags(implode(',', $this->_Languages)),
                                        'layout');
    }

    /**
     * Returns the items that should be displayed in the layout
     *
     * @access  public
     * @return  array   Items according to BASE_SCRIPT
     */
    function GetLayoutItems()
    {
        if (JAWS_SCRIPT == 'index') {
            return $this->_Model->GetLayoutItems();
        }
        $items = array();
        $items[] = array('id'            => null,
                         'gadget'        => '[REQUESTEDGADGET]',
                         'gadget_action' => '[REQUESTEDACTION]',
                         'display_when'  => '*',
                         'section'       => 'main',
                         );
        return $items;
    }

    /**
     * Is gadget item displayable?
     *
     * @access  public
     * @return  boolean
     */
    function IsDisplayable($gadget, $action, $display_when, $index)
    {
        $displayWhen = explode(',', $display_when);
        if ($display_when == '*' || ($index && in_array('index', $displayWhen))) {
            return true;
        }

        foreach ($displayWhen as $item) {
            $gActions = explode(';', $item);
            $g = array_shift($gActions);
            if ($g == $gadget) {
                if (empty($gActions) || in_array($action, $gActions)) {
                    return true;
                }
                break;
            }
        }

        return false;
    }

    /**
     * Look for the available gadgets and put them on the template
     *
     * @access  public
     */
    function Populate(&$goGadget, $am_i_index = false)
    {
        $this->_RequestedGadget = empty($goGadget)? '': $goGadget->GetName();
        $this->_RequestedAction = empty($goGadget)? '': $goGadget->GetAction();
        $items = $this->GetLayoutItems();
        if (is_array($items)) {
            $contents        = array();
            $gadgetsInLayout = array();
            foreach ($items as $item) {
                if (empty($this->_RequestedGadget) && ($item['gadget'] == '[REQUESTEDGADGET]')) {
                    continue;
                }

                if ($this->_Section != $item['section']) {
                    if (!empty($this->_Section)) {
                        $this->_Template->SetVariable('ELEMENT', $contentString);
                        $this->_Template->ParseBlock('layout/' . $this->_Section);
                        $this->_Section = '';
                    }
                    if (!$this->_Template->BlockExists('layout/' . $item['section'])) {
                        continue;
                    }
                    $this->_Section = $item['section'];
                    $this->_Template->SetBlock('layout/' . $this->_Section);
                    $this->_SectionAttributes = $this->_Template->GetCurrentBlockAttributes();
                    $currentContent = $this->_Template->GetCurrentBlockContent();
                    $this->_Template->SetCurrentBlockContent('{ELEMENT}');
                    $contentString  = '';
                }

                if ($this->IsDisplayable($this->_RequestedGadget,
                                         $this->_RequestedAction,
                                         $item['display_when'],
                                         $am_i_index))
                {
                    if (!isset($contents[$item['section']])) {
                        $contents[$item['section']] = array();
                    }

                    $content = '';
                    if ($item['gadget'] == '[REQUESTEDGADGET]') {
                        $gadgetsInLayout[$goGadget->GetName()] = $goGadget->GetName();
                        // FIXME: We need to use PutGadgetByParameter...
                        if (Jaws_Gadget::IsGadgetUpdated($goGadget->GetName())) {
                            $content = $this->PutGadget($goGadget->GetName(), $goGadget->GetAction(), $item['section'], true);
                        } elseif (isset($GLOBALS['log'])) {
                            $GLOBALS['log']->Log(JAWS_LOG_ERR, 'Trying to populate ' . $goGadget->GetName() .
                                                 ' in layout, but looks that it is not installed/upgraded');
                        }
                    } else {
                        $gadgetsInLayout[$item['gadget']] = $item['gadget'];
                        if (Jaws_Gadget::IsGadgetUpdated($item['gadget'])) {
                            $content = $this->PutGadget($item['gadget'], $item['gadget_action'], $item['section']);
                        }  elseif (isset($GLOBALS['log'])) {
                            $GLOBALS['log']->Log(JAWS_LOG_ERR, 'Trying to populate ' . $item['gadget'] .
                                                    ' in layout, but looks that it is not installed/upgraded');
                        }
                    }
                    if (!empty($content)) {
                        $contentString .= str_replace('{ELEMENT}', $content, $currentContent)."\n\n\n";
                    }
                }
            }
            if (!empty($this->_Section)) {
                $this->_Template->SetVariable('ELEMENT', $contentString);
                $this->_Template->ParseBlock('layout/' . $this->_Section);
            }
        }
    }

    /**
     * Put a gadget on the template
     *
     * @access  public
     * @param   string  $gadget  Gadget to put
     * @param   string  $action  Action to execute
     * @param   string  $section Where to put it
     * @param   boolean $requested Requested gadget (to know if we should load the LayoutHTML file or not)
     */
    function PutGadget($gadget, $action, $section, $requested = false)
    {
        $enabled = $GLOBALS['app']->Registry->Get('/gadgets/' . $gadget . '/enabled');
        if (Jaws_Error::isError($enabled)) {
            $enabled = 'false';
        }

        $output = '';
        if ($enabled == 'true') {
            if (JAWS_SCRIPT == 'admin') {
                $this->AddHeadLink('gadgets/'.$gadget.'/resources/style.css',
                                   'stylesheet',
                                   'text/css',
                                   '',
                                   null,
                                   true);
                $goGadget = $GLOBALS['app']->loadGadget($gadget, 'AdminHTML');
                if (!Jaws_Error::isError($goGadget)) {
                    $goGadget->SetAction($action);
                    //$this->_Template->SetVariable('ELEMENT', $goGadget->Execute());
                    $output = $goGadget->Execute();
                } else {
                    //$this->_Template->SetVariable('ELEMENT', '');
                }
            } else {
                if ($requested === true) {
                    $goGadget = $GLOBALS['app']->loadGadget($gadget, 'HTML');
                    $goGadget->SetAction($action);
                    //$this->_Template->SetVariable('ELEMENT', $goGadget->Execute());
                    $output = $goGadget->Execute();
                } else {
                    preg_match_all('/^([a-z0-9]+)\((.*?)\)$/i', $action, $matches);
                    if (isset($matches[1][0]) && isset($matches[2][0])) {
                        $action = $matches[1][0];
                        $params = $matches[2][0];
                    }

                    $goGadget = $GLOBALS['app']->loadGadget($gadget, 'LayoutHTML');
                    if (!Jaws_Error::isError($goGadget)) {
                        $GLOBALS['app']->Registry->LoadFile($gadget);
                        if (method_exists($goGadget, $action)) {
                            $output = isset($params)? $goGadget->$action($params) : $goGadget->$action();
                        } elseif (isset($GLOBALS['log'])) {
                            $GLOBALS['log']->Log(JAWS_LOG_ERR, "Action $action in $gadget's LayoutHTML dosn't exist.");
                        }
                    } else {
                        //$this->_Template->SetVariable('ELEMENT', '');
                        if (isset($GLOBALS['log'])) {
                            $GLOBALS['log']->Log(JAWS_LOG_ERR, $gadget ." is missing the LayoutHTML. Jaws can't execute Layout " .
                                                 "actions if the file doesn't exists");
                        }
                    }
                }
            }
        } else {
            Jaws_Error::Fatal('Gadget ' . $gadget . ' is not enabled', __FILE__, __LINE__);
        }
        if (Jaws_Error::isError($output)) {
            if (isset($GLOBALS['log'])) {
                $GLOBALS['log']->Log(JAWS_LOG_ERR, 'In '.$gadget.'::'.$action.','.$output->GetMessage());
            }
            return '';
        }

        return $output;
    }

    /**
     * Get the HTML code of the head content.
     *
     * @access  public
     */
    function GetHeaderContent(&$headLink, &$headScript, &$headMeta, &$headOther)
    {
        $headContent = '';
        // meta
        foreach ($headMeta as $meta) {
            if ($meta['use_http_equiv']) {
                $meta_add = 'http-equiv="' . $meta['name'] . '"';
            } else {
                $meta_add = 'name="' . $meta['name'] . '"';
            }

            $headContent.= '<meta ' . $meta_add . ' content="' . $meta['content'] . '" />' . "\n";
        }

        // link
        foreach ($headLink as $link) {
            $title = '';
            $headContent.= '<link rel="' . $link['rel'] . '"';
            if (!empty($link['media'])) {
                $headContent.= ' media="' . $link['media'] . '"';
            }
            if (!empty($link['type'])) {
                $headContent.= ' type="' . $link['type'] . '"';
            }
            if (!empty($link['href'])) {
                $headContent.= ' href="' . $link['href'] . '"';
            }
            if (!empty($link['title'])) {
                $headContent.= ' title="' . $link['title'] . '"';
            }
            $headContent.= ' />' . "\n";
        }

        //script
        foreach ($headScript as $link) {
            $headContent.= '<script type="' . $link['type'] . '" src="' . $link['href'] . '"></script>' . "\n";
        }

        // other
        foreach ($headOther as $element) {
            $headContent .= $element . "\n";
        }

        return $headContent;
    }

    /**
     * Shows the HTML of the Layout.
     *
     * @access  public
     */
    function Show($res_echo = true)
    {
        $headContent = $this->GetHeaderContent($this->_HeadLink, $this->_ScriptLink, $this->_HeadMeta, $this->_HeadOther);

        if (!empty($headContent)) {
            $this->_Template->SetBlock('layout/head');
            $this->_Template->SetVariable('ELEMENT', $headContent);
            $this->_Template->ParseBlock('layout/head');
        }

        if (JAWS_SCRIPT == 'index') {
            $this->PutTitle();
            $this->PutMetaKeywords();
            $this->PutMetaLanguages();
        }

        // parse template an show the HTML
        $this->_Template->ParseBlock('layout');
        if ($res_echo) {
            $content = $this->_Template->Get();
            if ($GLOBALS['app']->GZipEnabled()) {
                $content = gzencode($content, COMPRESS_LEVEL, FORCE_GZIP);
                header('Content-Length: '.strlen($content));
                header('Content-Encoding: '.(strpos($GLOBALS['app']->GetBrowserEncoding(), 'x-gzip')!== false? 'x-gzip' : 'gzip'));
            }
            echo $content;
        } else {
            return $this->_Template->Get();
        }
    }

    /**
     * Add a meta tag
     *
     * @access  public
     * @param   string  $name           Key of the meta tag
     * @param   string  $content        Value of the key
     * @param   boolean $use_http_equiv Use the equiv of HTTP
     */
    function AddHeadMeta($name, $content, $use_http_equiv = false)
    {
        $this->_HeadMeta[$name]['name']    = $name;
        $this->_HeadMeta[$name]['content'] = $content;
        $this->_HeadMeta[$name]['use_http_equiv'] = $use_http_equiv;
    }

    /**
     * Add a HeadLink
     *
     * @access  public
     * @param   string  $href  The HREF
     * @param   string  $rel   The REL that will be associated
     * @param   string  $type  Type of HeadLink
     * @param   boolean $checkInTheme Check if resource exists in the current theme directory
     * @param   string  $title Title of the HeadLink
     * @param   string  $media Media type, screen, print or such
     * @param   boolean $standanlone for use in static load
     */
    function AddHeadLink($href, $rel, $type, $title = '', $direction = null, $checkInTheme = false,
                         $media = 'screen', $standanlone = false)
    {
        $fileName = basename($href);
        $fileExt  = strrchr($fileName, '.');
        $fileName = substr($fileName, 0, -strlen($fileExt));
        if (substr($href, 0, 1) == '/') {
            $path = substr($href , 1, - strlen($fileName.$fileExt));
        } else {
            $path = substr($href , 0, - strlen($fileName.$fileExt));
        }

        $brow = $GLOBALS['app']->GetBrowserFlag();
        $brow = empty($brow)? '' : '.'.$brow;

        $prefix = '.' . strtolower(empty($direction) ? _t('GLOBAL_LANG_DIRECTION') : $direction);
        if ($prefix !== '.rtl') {
            $prefix = '';
        }

        // First we try to load the css files from the theme dir.
        if ($checkInTheme) {
            $themeDir = JAWS_DATA . 'themes/' . $GLOBALS['app']->GetTheme() . '/';
            $gadget = str_replace(array('gadgets/', 'resources/'), '', $path);
            $href = $themeDir . $gadget . $fileName . $prefix . $fileExt;
            if (!empty($prefix) && !file_exists($href)) {
                $href = JAWS_PATH . 'gadgets/' . $gadget . 'resources/' . $fileName . $prefix . $fileExt;
                if (!file_exists($href)) {
                    $href = $themeDir . $gadget . $fileName . $fileExt;
                }
            }

            if (!file_exists($href)) {
                $href = JAWS_PATH . 'gadgets/' . $gadget . 'resources/' . $fileName . $fileExt;
                if (!file_exists($href)) {
                    return false;
                }
            }
            $href = JAWS_DPATH . str_replace(JAWS_PATH, '', $href);
        } else {
            $path = (strpos($path, '://') === false)? (JAWS_DPATH . $path) : $path;
            $href = $path . $fileName . $prefix . $fileExt;
            if (!empty($prefix) && !file_exists($href)) {
                $href = $path . $fileName . $fileExt;
            }
        }

        $hLinks[] = array(
            'href'  => $href,
            'rel'   => $rel,
            'type'  => $type,
            'title' => $title,
            'media' => $media,
        );
        if (!$standanlone) $this->_HeadLink[] = $hLinks[0];

        $brow_href = substr_replace($href, $brow, strrpos($href, '.'), 0);
        if (!empty($brow) && file_exists($brow_href)) {
            $hLinks[] = array(
                'href'  => $brow_href,
                'rel'   => $rel,
                'type'  => $type,
                'title' => $title,
                'media' => $media,
            );
            if (!$standanlone) $this->_HeadLink[] = $hLinks[1];
        }

        return $standanlone? $hLinks : true;
    }

    /**
     * Add a Javascript source
     *
     * @access  public
     * @param   string  $href   The path for the source.
     * @param   string  $type   The mime type.
     * @param   boolean $standanlone for use in static load
     * @return  null
     * @since   0.6
     */
    function AddScriptLink($href, $type = 'text/javascript', $standanlone = false)
    {
        $sLink = array(
                'href' => (substr($href,0,1)=='?' || strpos($href,'://')!==false)? $href : (JAWS_DPATH . $href),
                'type' => $type
        );
        if (!$standanlone) $this->_ScriptLink[] = $sLink;

        return $standanlone? $sLink : true;
    }

    /**
     * Add other info to the head tag
     *
     * @access  public
     * @param   string  $text Text to add.
     * @return  null
     * @since   0.6
     */
    function addHeadOther($text)
    {
        $this->_HeadOther[] = $text;
    }

    /**
     * Get Requested gadget
     */
    function GetRequestedGadget()
    {
        return $this->_RequestedGadget;
    }
}
?>
