<?php
/**
 * Class that deals like a wrapper between Jaws and pear/Mail
 *
 * @category   Mail
 * @package    Core
 * @author     David Coallier <davidc@agoraproduction.com>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2005-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
class Jaws_Mail
{
    // {{{ Variables
    /**
     * The mailer type
     * @param string $mailer The mailer type
     */
    var $mailer = '';

    // {{{ Variables
    /**
     * Send email via this email
     * @param string $from_email The from email
     */
    var $from_email = '';

    // {{{ Variables
    /**
     * From name
     * @param string $from_name The from nane
     */
    var $from_name = '';

    // {{{ Variables
    /**
     * The server infos (host,login,pass)
     * @param array $server The server infos
     */
    var $params = array();

    /**
     * The email recipients.
     * @param array $recipients The recipients.
     */
    var $recipient = array();

    /**
     * The email headers
     *
     * @param array string $headers The headers of the mail.
     */
    var $headers = array();

    /**
     * The crlf character(s)
     *
     * @param string $crlf
     */
    var $crlf = "\n";

    /**
     * A object of Mail_Mime
     *
     * @param object $mail_mime
     */
    var $mail_mime;

    /**
     * This creates the mail object that will
     * add recipient, send emails to destinated
     * email addresses calling functions.
     *
     * @access constructor
     */
    function Jaws_Mail($init = true)
    {
        require_once 'Mail.php';
        require_once 'Mail/mime.php';
        require_once JAWS_PATH . 'include/Jaws/UTF8.php';
        $this->mail_mime =& new Mail_Mime($this->crlf);

        if ($init) {
            $this->Init();
        }
    }

    /**
     * This function loads the mail settings from
     * the registry.
     *
     * @access public
     */
    function Init()
    {
        if (!isset($GLOBALS['app'])) {
            return new Jaws_Error('$GLOBALS[\'app\'] not available', 'MAIL', JAWS_ERROR_ERROR);
        }

        // Get the Mail settings data from Registry
        $this->mailer     = $GLOBALS['app']->Registry->Get('/network/mailer');
        $this->from_email = $GLOBALS['app']->Registry->Get('/network/from_email');
        $this->from_name  = $GLOBALS['app']->Registry->Get('/network/from_name');

        $params = array();
        $params['sendmail_path'] = $GLOBALS['app']->Registry->Get('/network/sendmail_path');
        $params['sendmail_args'] = $GLOBALS['app']->Registry->Get('/network/sendmail_args');
        $params['host']          = $GLOBALS['app']->Registry->Get('/network/smtp_host');
        $params['port']          = $GLOBALS['app']->Registry->Get('/network/smtp_port');
        $params['auth']          = $GLOBALS['app']->Registry->Get('/network/smtp_auth')  == 'true';
        $params['pipelining']    = $GLOBALS['app']->Registry->Get('/network/pipelining') == 'true';
        $params['username']      = $GLOBALS['app']->Registry->Get('/network/smtp_user');
        $params['password']      = $GLOBALS['app']->Registry->Get('/network/smtp_pass');

        $this->params = $params;
        return $this->params;
    }

    /**
     * This adds a recipient to the mail to send.
     *
     * @param string $recipient  The recipient to add.
     * @param string $valid      Do we validate the email ?
     * @param bool $checkdns   Do we check the MX record ?
     * @access public
     * @return string recipients
     */
    function AddRecipient($recipient, $valid = true, $checkdns = false)
    {
        if (trim($recipient) !== '') {
            if (!$valid) {
                return $this->recipient[] = $recipient;
            }

            require_once 'Validate.php';
            if (!Validate::email($recipient, $checkdns)) {
                return new Jaws_Error("Email not valid", 'MAIL', JAWS_ERROR_ERROR);
            }

            return $this->recipient[] = $recipient;
        }
    }

    /**
     * This function sets the headers of the email to send.
     *
     * @param string $to       Send to.
     * @param string $from     Who the email is from.
     * @param string $subject  Subject of the email.
     * @access protected
     * @return array string headers
     */
    function SetHeaders($to = '', $from_name = '', $from_email = '', $subject = '')
    {
        /**
         * From, To, subject
         */
        $params       = array();
        $params['To'] = $to;

        if (!empty($this->from_email)) {
            $params['Subject'] = $from_name . ' <' . $from_email . '> : ' . $subject;
            $from_name  = $this->from_name;
            $from_email = $this->from_email;
        } else {
            $params['Subject'] = $subject;
        }

        if ($this->mailer == 'phpmail') {
            $params['From'] = $from_email;
        } else {
            $params['From'] = Jaws_UTF8::encode_mimeheader($from_name) . ' <'.$from_email.'>';
        }

        return $this->headers = $params;
    }

    /**
     * This function returns the set headers.
     *
     * @access public
     * @return $this->headers
     */
    function GetHeaders()
    {
        return $this->headers;
    }

    /**
     * This function sets the body, the structure
     * of the email, what's in it..
     *
     * @param string $body   The body of the email
     * @param string $format The format to use.
     * @access protected
     * @return string $body
     */
    function SetBody($body, $format = 'text')
    {
        if (!isset($body) && empty($body)) {
            return false;
        }

        switch ($format) {
            case 'file':
                $res = $this->mail_mime->addAttachment($body);
                break;
            case 'image':
                $res = $this->mail_mime->addHTMLImage($body);
                break;
            case 'html':
                $res = $this->mail_mime->setHTMLBody($body);
                break;
            case 'text':
                $res = $this->mail_mime->setTXTBody($body);
                break;
            default:
                $res = false;
        }

        return $res;
    }

    /**
     * This function sends the email
     *
     * @param array string recipients The recipients
     * @param array string headers    The email headers
     * @param       string from       The email sender
     * @param       string body       The email body
     * @access public
     */
    function send()
    {
        $mail = null;
        switch ($this->mailer) {
            case 'phpmail':
                $mail =& Mail::factory('mail');
                break;
            case 'sendmail':
                $mail =& Mail::factory('sendmail', $this->params);
                break;
            case 'smtp':
                $mail =& Mail::factory('smtp', $this->params);
                break;
            default:
                return false;
        }

        $realbody = $this->mail_mime->get(array('html_encoding' => '8bit',
                                     'text_encoding' => '8bit',
                                     'head_encoding' => 'base64',
                                     'html_charset'  => 'utf-8',
                                     'text_charset'  => 'utf-8',
                                     'head_charset'  => 'utf-8',
                                    ));

        $headers  = $this->mail_mime->headers($this->headers);

        $e = $mail->send($this->recipient, $headers, $realbody);
        if (PEAR::isError($e)) {
            return new Jaws_Error($e->getMessage());
        }

        return true;
    }

    /**
     * Resets the values and updates
     *
     * @access  public
     */
    function ResetValues()
    {
        $this->headers   = array();
        $this->recipient = array();
        unset($this->mail_mime);
        $this->mail_mime =& new Mail_Mime($this->crlf);
    }

}
