<?php
/**
 * Class to manage User session.
 *
 * @category   Session
 * @package    Core
 * @author     Ivan -sk8- Chavero <imcsk8@gluch.org.mx>
 * @author     Jonathan Hernandez <ion@suavizado.com>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2004-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
define('SESSION_ERROR_SYNC', "Can't sync session");
define('SESSION_ERROR_MULTSESSIONS', "Multiple sessions in DB");
define('SESSION_RESERVED_ATTRIBUTES', "session_id,type,user_id,updatetime,username,logged,user_type,acl");

/**
 * Responses
 */
define('RESPONSE_WARNING', 'RESPONSE_WARNING');
define('RESPONSE_ERROR',   'RESPONSE_ERROR');
define('RESPONSE_NOTICE',  'RESPONSE_NOTICE');

class Jaws_Session
{
    /**
     * Authentication method
     * @var     string $_AuthMethod
     * @access  private
     */
    var $_AuthMethod;

    /**
     * Logged flag
     * @var     boolean $_Logged
     * @access  private
     * @see     Logged()
     */
    var $_Logged;

    /**
     * Last error message
     * @var     string $_Error
     * @access  private
     * @see     GetError()
     */
    var $_Error;

    /**
     * Attributes array
     * @var     array $_Attributes
     * @access  private
     * @see     SetAttribute(), GetAttibute()
     */
    var $_Attributes = array();

    /**
     * Changes flag
     * @var     boolean $_HasChanged
     * @access  private
     */
    var $_HasChanged;

    /**
     * session unique identifier
     * @var     string $_SessionID
     * @access  private
     */
    var $_SessionID;

    /**
     * Session is only for admins
     *
     * @access  public
     * @var     boolean
     */
    var $_OnlyAdmins = false;

    /**
     * An interface for available drivers
     *
     * @access  public
     */
    function &factory()
    {
        $SessionType = ucfirst(strtolower(APP_TYPE));
        $sessionFile = JAWS_PATH . 'include/Jaws/Session/'. $SessionType .'.php';
        if (!file_exists($sessionFile)) {
            if (isset($GLOBALS['log'])) {
                $GLOBALS['log']->Log(JAWS_LOG_DEBUG, "Loading session $SessionType failed.");
            }
            return new Jaws_Error("Loading session $SessionType failed.", 'SESSION', JAWS_ERROR_ERROR);
        }

        include_once $sessionFile;
        $className = 'Jaws_Session_' . $SessionType;
        $obj = new $className();
        return $obj;
    }

    /**
     * Initializes the Session
     */
    function Init()
    {
        $this->_AuthMethod = $GLOBALS['app']->Registry->Get('/config/auth_method');
        $authFile = JAWS_PATH . 'include/Jaws/AuthScripts/' . $this->_AuthMethod . '.php';
        if (empty($this->_AuthMethod) || !file_exists($authFile)) {
            if (isset($GLOBALS['log'])) {
                $GLOBALS['log']->Log(JAWS_LOG_DEBUG, $method . ' Error: ' . $authFile . ' file doesn\'t exists, using DefaultAuth');
            }
            $this->_AuthMethod = 'DefaultAuthentication';
        }

        // Try to restore session...
        $this->_HasChanged = false;

        // Load cache
        include_once JAWS_PATH . 'include/Jaws/Session/Cache.php';
        $this->_cache = new Jaws_Session_Cache;

        // Delete expired sessions
        $last_expired_file = JAWS_DATA . 'last_expired';
        $last_expired = @file_get_contents($last_expired_file);
        if ($last_expired === false || 
           ($last_expired < (time() - ($GLOBALS['app']->Registry->Get('/policy/session_idle_timeout') * 60))))
        {
            Jaws_Utils::file_put_contents($last_expired_file, time());
            $this->_cache->DeleteExpiredSessions();
        }
    }

    /**
     * Set the session type for admins
     *
     * @access  public
     */
    function OnlyAdmins()
    {
        $this->_OnlyAdmins = true;
    }

    /**
     * Gets the session mode
     *
     * @access  public
     * @return  string  Session mode
     */
    function GetMode()
    {
        return $this->_Mode;
    }

    /**
     * Login
     *
     * @param   string  $username Username
     * @param   string  $password Password
     * @param   boolean $remember Remember me
     * @return  boolean True if succeed.
     */
    function Login($username, $password, $remember)
    {
        if (isset($GLOBALS['log'])) {
            $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'LOGGIN IN');
        }

        if ($username !== '' && $password !== '') {
            require_once JAWS_PATH . 'include/Jaws/AuthScripts/' . $this->_AuthMethod . '.php';
            $authFunc = $this->_AuthMethod;
            $result = $authFunc($username, $password, $this->_OnlyAdmins);
            if (!Jaws_Error::isError($result)) {
                if (!empty($this->_SessionID)) {
                    $this->Logout();
                }

                $this->Create($username, $remember);
                $this->_Logged = true;

                // Update login time
                $user_id = $this->GetAttribute('user_id');
                $userModel = new Jaws_User;
                $userModel->updateLoginTime($user_id);

                return true;
            }

            return $result;
        }

        return new Jaws_Error(_t('GLOBAL_ERROR_LOGIN_WRONG'));
    }

    /**
     * Return session login status
     * @access  public
     */
    function Logged()
    {
        //Can non-admins be logged?
        if ($this->_OnlyAdmins === true) {
            $user_type = $this->GetAttribute('user_type');
            //Only admins eh??.. so if you are not an admin you are not logged..
            if ($user_type == 2) {
                return false;
            }
        }
        return $this->_Logged;
    }

    /**
     * Logout
     *
     * Logout from session
     * delete session from the system
     */
    function Logout()
    {
        $this->SetAttribute('logged', false);
        $this->_cache->Delete($this->_SessionID);
        $this->_SessionID = '';
        $this->_Attributes = array();
        if (isset($GLOBALS['log'])) {
            $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'Session succesfully destroyed');
        }
    }

    /**
     * Return last error message
     * @access  public
     */
    function GetError()
    {
        return $this->_Error;
    }

    /**
     * Loads Jaws Session
     *
     * @param   string $session_id Session ID
     * @return  boolean True if can load session, false if not
     */
    function Load($session_id)
    {
        if (isset($GLOBALS['log'])) {
            $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'Loading session');
        }

        $session = $this->_cache->GetSession($session_id);
        if (is_array($session)) {
            $md5 = md5($session['user_id'] . $session['hash']);
            if ($md5 === $session['md5']) {
                $this->_Attributes = unserialize($session['hash']);
                $this->_SessionID = $session_id;
                if (isset($GLOBALS['log'])) {
                    $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'Session was OK');
                }
            } else {
                if (isset($GLOBALS['log'])) {
                    $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'Session found but checksum fail');
                }
            }
        } else {
            if (isset($GLOBALS['log'])) {
                $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'No previous session exists');
            }
        }

        return !empty($this->_SessionID);
    }

    /**
     * Create a new session for a given username
     * @param   string  $username Username
     * @param   boolean $remember Remember me
     * @return  boolean True if can create session.
     */
    function Create($username, $remember = false)
    {
        $info = array();
        if (empty($username)) {
            $info['id']        = 0;
            $info['username']  = '';
            $info['user_type'] = 2;
            $info['name']      = '';
            $info['email']     = '';
            $info['url']       = '';
            $info['language']  = '';
            $info['theme']     = '';
            $info['editor']    = '';
            $info['timezone']  = null;
            $groups = array();
        } else {
            require_once JAWS_PATH . 'include/Jaws/User.php';
            $userModel = new Jaws_User;
            $info = $userModel->GetInfoByUsername($username);
            if (Jaws_Error::IsError($info) || !isset($info['username'])) {
                return false;
            }

            $groups = $userModel->GetGroupsOfUser($info['id']);
            if ($groups === false) {
                return false;
            }
        }
        $session_id = md5(uniqid(rand(), true)) . time() . floor(microtime()*1000);
        //Don't know if session_id, user_id and modification time
        //should be on the _Attributes array since they are part of
        // the structure of session and session_user_data tables
        //session_id stays in the hash but only for knowing wich session
        //was the last to alter the session_user_data table.
        $this->_SessionID = $session_id;
        /**
         * We need to make sure Attributes is clean cause there's the possibility
         * that user was logged as another user (anonymous for example) with different
         * attributes (first Load check WebSession)
         */
        $this->_Attributes = array();
        $this->SetAttribute('logged', !empty($username));
        $this->SetAttribute('session_id', $session_id);
        $this->SetAttribute('user_id', $info['id']);
        $this->SetAttribute('user_type', (int)$info['user_type']);
        $this->SetAttribute('type', APP_TYPE);
        $this->SetAttribute('life_time', $remember?
                            (int)$GLOBALS['app']->Registry->Get('/policy/session_remember_timeout')*3600 : 0);
        $this->SetAttribute('updatetime', time());
        $this->SetAttribute('username', $info['username']);

        if (isset($info['last_login'])) {
            $this->SetAttribute('last_login', $info['last_login']);
        } else {
            MDB2::loadFile('Date');
            $this->SetAttribute('last_login', MDB2_Date::mdbNow());
        }

        //profile
        $this->SetAttribute('name',  $info['name']);
        $this->SetAttribute('email', $info['email']);
        $this->SetAttribute('url',   $info['url']);

        //preferences
        $this->SetAttribute('language', $info['language']);
        $this->SetAttribute('theme',    $info['theme']);
        $this->SetAttribute('editor',   $info['editor']);
        $this->SetAttribute('timezone', $info['timezone']);

        $groups = array_map(create_function('$row','return $row["group_id"];'), $groups);
        $this->SetAttribute('groups', $groups);

        if ($this->Synchronize()) {
            return true;
        }

        return false;
    }

    /**
     * Reset current session
     * @return  boolean True if can reset it
     */
    function Reset()
    {
        $username = $this->GetAttribute('username');
        $this->_Attribute = array();
        $this->SetAttribute('session_id', $this->_SessionID);
        $this->SetAttribute('type', APP_TYPE);
        $this->SetAttribute('updatetime', time());
        $this->SetAttribute('username', $username);
        $this->SetAttribute('logged', false);
        $this->SetAttribute('user_type', 0);
        $this->SetAttribute('groups',   '');
        $this->SetAttribute('name',     '');
        $this->SetAttribute('email',    '');
        $this->SetAttribute('url',      '');
        $this->SetAttribute('language', '');
        $this->SetAttribute('theme',    '');
        $this->SetAttribute('editor',   '');
        $this->SetAttribute('timezone', null);
        if ($this->Synchronize()) {
            return true;
        }
        return false;
    }

    /**
     * Set a session attribute
     *
     * @param   string $name attribute name
     * @param   string $value attribute value
     * @return  boolean True if attribute has changed
     */
    function SetAttribute($name, $value)
    {
        if (
            !isset($this->_Attributes[$name]) ||
            (isset($this->_Attributes[$name]) && $this->_Attributes[$name] != $value)
        ) {
            if (is_array($value) && $name == 'LastResponses') {
                $this->_Attributes['LastResponses'][] = $value;
            } else {
                $this->_Attributes[$name] = $value;
            }
            $this->_HasChanged = true;
            return true;
        }

        return false;
    }

    /**
     * Get a session attribute
     *
     * @param   string $name attribute name
     * @return  string value of the attribute
     */
    function GetAttribute($name)
    {
        if (array_key_exists($name, $this->_Attributes)) {
            return $this->_Attributes[$name];
        }

        return null;
    }

    /**
     * Delete a session attribute
     *
     * @param   string $name attribute name
     * @return  boolean True if attribute has been deleted
     */
    function DeleteAttribute($name)
    {
        if (array_key_exists($name, $this->_Attributes)) {
            unset($this->_Attributes[$name]);
            $this->_HasChanged = true;
            return true;
        }

        return false;
    }

    /**
     * Get permission on a given gadget/task
     *
     * @param   string $gadget Gadget name
     * @param   string $task Task name
     * @return  boolean True if granted, else False
     */
    function GetPermission($gadget, $task)
    {
        if ($this->IsAdmin() === false) {
            return false;
        }

        if ($this->IsSuperAdmin() === true) {
            return true;
        }

        $groups = $this->GetAttribute('groups');
        $user = $this->GetAttribute('username');
        return $GLOBALS['app']->ACL->GetFullPermission($user, $groups, $gadget, $task);
    }

    /**
     * Check permission on a given gadget/task
     *
     * @param   string $gadget Gadget name
     * @param   string $task Task name
     * @param   string $errorMessage Error message to return
     * @return  boolean True if granted, else throws an Exception(Jaws_Error::Fatal)
     */
    function CheckPermission($gadget, $task, $errorMessage = '')
    {
        if ($this->GetPermission($gadget, $task)) {
            return true;
        }

        if (empty($errorMessage)) {
            $errorMessage = 'User '.$this->GetAttribute('username').
                ' don\'t have permission to execute '.$gadget.'::'.$task;
        }

        Jaws_Error::Fatal($errorMessage, __FILE__, __LINE__);
    }

    /**
     * Returns true if user is a super-admin (aka superroot)
     *
     * @access  public
     * @return  boolean
     */
    function IsSuperAdmin()
    {
        if ($this->GetAttribute('user_type') === 0) {
            return true;
        }
        return false;
    }

    /**
     * Returns true if user is an admin (or super-admin)
     *
     * @access  public
     * @return  boolean
     */
    function IsAdmin()
    {
        if ($this->GetAttribute('user_type') <= 1) {
            return true;
        }
        return false;
    }

    function Synchronize()
    {
        if (isset($GLOBALS['log'])) {
            $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'Synchronizing session');
        }

        $res = $this->_cache->Synchronize();
        if ($res) {
            if (isset($GLOBALS['log'])) {
                $GLOBALS['log']->Log(JAWS_LOG_DEBUG, 'New session created');
            }
        }

        return $res;
    }

    /**
     * Push a simple response (no CSS and special data)
     *
     * @access  public
     * @param   string  $msg    Response's message
     */
    function PushSimpleResponse($msg, $resource = 'SimpleResponse')
    {
        $this->SetAttribute($resource, $msg);
    }

    /**
     * Prints (returns) the last simple response
     *
     * @access  public
     * @return  string  Last simple response
     */
    function PopSimpleResponse($resource = 'SimpleResponse')
    {
        $response = $this->GetAttribute($resource);
        $this->DeleteAttribute($resource);
        if (empty($response)) {
            return false;
        }
        return $response;
    }

    /**
     * Add the last response to the session system
     *
     * @access  public
     * @param   string  $gadget Gadget's name
     * @param   string  $msg    Response's message
     */
    function PushLastResponse($msg, $level = RESPONSE_WARNING)
    {
        if (!defined($level)) {
            $level = RESPONSE_WARNING;
        }

        switch ($level) {
            case RESPONSE_ERROR:
                $css = 'error-message';
                break;
            case RESPONSE_NOTICE:
                $css = 'notice-message';
                break;
            case RESPONSE_WARNING:
            default:
                $css = 'warning-message';
                break;
        }

        $this->SetAttribute('LastResponses',
                            array(
                                  'message' => $msg,
                                  'level'   => $level,
                                  'css'     => $css
                                  )
                            );
    }

    /**
     * Prints and deletes the last response of a gadget
     *
     * @access  public
     * @param   string  $gadget Gadget's name
     * @return  string  Returns the message of the last response and false if there's no response
     */
    function PopLastResponse()
    {
        $responses = $this->GetAttribute('LastResponses');
        if ($responses === null) {
            return false;
        }

        $this->DeleteAttribute('LastResponses');
        $responses = array_reverse($responses);
        if (empty($responses[0]['message'])) {
            return false;
        }

        return $responses;
    }
}
