<?php
/**
 * Class to deal with HTML string
 *
 * @category   JawsType
 * @package    Core
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2005-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
class Jaws_StringHtml
{
    /**
     * Return the string with the valid HTML tags
     * Of course this doesn't strip out id="" class="" and similar
     *
     * @param  string  $string  Input string with lot of HTML tags
     * @return string  Clean string, with just some allowed tags
     * @access public
     */
    function WithHTML($string)
    {
        $allowed_tags = '<a><b><i><u><br><pre>';
        $string = strip_tags($string, $allowed_tags);
        return $string;
    }

    /**
     * Inserts HTML line breaks before all newlines in not empty strings
     *
     * @param   string  $text  Text to parse
     * @return  string
     * @access  public
     */
    function html_nl2br($text)
    {
        $text = str_replace('\"', '"', $text);
        return (strlen(trim($text, "\t\n\r\0\x0B")) == 0)? '' : nl2br($text);
    }

    /**
     * Parses the text, adding paragraph tags when is needed
     *
     * @param   string  $text  Text to parse
     * @return  string  The parsed text
     * @access  public
     * Copied FROM: http://photomatt.net/scripts/autop and other sites that are using autop
     */
    function AutoParagraph($text)
    {
        $text = preg_replace('|<br />\s*<br />|', "\n\n", $text);
        $text = preg_replace('!(<(?:table|math|ul|ol|li|form|blockquote|h[1-6])[^>]*>)!', "\n$1", $text); // Space things out a little
        $text = preg_replace('!(</(?:table|math|ul|ol|li|form|blockquote|h[1-6])>)!', "$1\n", $text); // Space things out a little
        $text = preg_replace("/(\r\n|\r)/", "\n", $text); // cross-platform newlines
        $text = preg_replace("/\n\n+/", "\n\n", $text); // take care of duplicates

        //$text = preg_replace('/\n?(.+?)(?:\n\s*\n|\z)/s', "\t<p>$1</p>\n", $text); // make paragraphs, including one at the end
        $text = preg_replace('|<p>\s*?</p>|', '', $text); // under certain strange conditions it could create a P of entirely whitespace
        $text = preg_replace("|<p>(<li.+?)</p>|", "$1", $text); // problem with nested lists
        $text = preg_replace('|<p><blockquote([^>]*)>|i', "<blockquote$1><p>", $text);
        $text = str_replace('</blockquote></p>', '</p></blockquote>', $text);
        $text = preg_replace('!<p>\s*(</?(?:table|math|tr|td|th|div|ul|ol|li|select|form|blockquote|p|h[1-6])[^>]*>)!', "$1", $text);
        $text = preg_replace('!(</?(?:table|math|tr|td|th|div|ul|ol|li|select|form|blockquote|p|h[1-6])[^>]*>)\s*</p>!', "$1", $text);

        // replace line breaks(\n) with html <br /> tag
        $text = '<jaws>'.$text.'</jaws>'; // Add custom tag for regular expression work correctly
        $text = preg_replace("/(>\n|>)([^<]*)(<\/?)/ie", "'$1'.Jaws_StringHtml::html_nl2br('$2').'$3'", $text);
        $text = substr($text, 6, -7); // remove custom tag

        $text = preg_replace('!(</?(?:table|math|tr|td|th|div|dl|dd|dt|ul|ol|li|select|form|blockquote|p|h[1-6])[^>]*>)\s*<br />!',
                             "$1", $text);
        $text = preg_replace('!<br />(\s*</?(?:p|math|li|div|th|td|ul|ol)>)!', '$1', $text);
        $text = preg_replace('/&([^#])(?![a-z]{1,8};)/', '&#038;$1', $text);

        $text = preg_replace_callback("/<(math|pre)[^>]*>(.*?)<\/(math|pre)>/s",
                                      array('Jaws_StringHtml', 'CleanPRETags'),
                                      $text);

        return $text;
    }

    /**
     * Deletes <br />'s in certain cases (math, pre, etc..)
     *
     * @access  public
     * @param   string  $text Text to replace
     * @return  stirng  Clean text (without <br />'s)
     */
    function CleanPRETags($text)
    {
        $text = str_replace(array('<p>', '</p>', '<br />'), '', $text[0]);
        return $text;
    }
}
