<?php
/**
 * This class is for Jaws_User table operations
 *
 * @category   User
 * @package    Core
 * @author     Ivan -sk8- Chavero <imcsk8@gluch.org.mx>
 * @author     Jonathan Hernandez <ion@suavizado.com>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2005-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
class Jaws_User
{
    /**
     * Validate a user
     *
     * @access  public
     * @param   string  $user      User to validate
     * @param   string  $password  Password of the user
     * @param   string  $onlyAdmin Only validate for admins
     * @return  boolean Returns true if the user is valid and false if not
     */
    function Valid($user, $password, $onlyAdmin = false)
    {
        $params         = array();
        $params['user'] = strtolower($user);
        $GLOBALS['db']->dbc->loadModule('Function', null, true);
        $username = $GLOBALS['db']->dbc->function->lower('[username]');

        $sql = "
            SELECT [id], [passwd], [user_type], [bad_passwd_count], [last_access], [enabled]
            FROM [[users]]
            WHERE $username = {user}";

        $result = $GLOBALS['db']->queryRow($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        if (isset($result['id'])) {
            if ($onlyAdmin && $result['user_type'] > 1) {
                return new Jaws_Error(_t('GLOBAL_ERROR_LOGIN_ONLY_ADMIN'));
            }

            if ($result['bad_passwd_count'] >= $GLOBALS['app']->Registry->Get('/policy/passwd_bad_count') &&
               ((time() - $result['last_access']) <= $GLOBALS['app']->Registry->Get('/policy/passwd_lockedout_time')))
            {
                return new Jaws_Error(_t('GLOBAL_ERROR_LOGIN_LOCKED_OUT'));
            }

            if ($result['passwd'] == md5($password)) {
                if (!$result['enabled']) {
                    return new Jaws_Error(_t('GLOBAL_ERROR_LOGIN_DISABLED'));
                }

                return array('id' => $result['id'],
                            'user_type' => $result['user_type']);

            } else {
                $params['id']          = $result['id'];
                $params['bad_count']   = $result['bad_passwd_count'] + 1;
                $params['last_access'] = time();

                $sql = '
                    UPDATE [[users]] SET
                        [last_access]      = {last_access},
                        [bad_passwd_count] = {bad_count}
                    WHERE [id] = {id}';

                $result = $GLOBALS['db']->query($sql, $params);
            }
        }

        return new Jaws_Error(_t('GLOBAL_ERROR_LOGIN_WRONG'));
    }

    /**
     * Updates the last login time for the given user
     *
     * @param $user_id integer user id of the user being updated
     * @return boolean true if all is ok, false if error
     */
    function updateLoginTime($user_id)
    {
        MDB2::loadFile('Date');
        $params               = array();
        $params['last_login'] = MDB2_Date::mdbNow();
        $params['id']         = (int)$user_id;

        $sql = '
            UPDATE [[users]] SET
                [last_login]       = {last_login},
                [bad_passwd_count] = 0
            WHERE [id] = {id}';
        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::isError($result)) {
            return false;
        }

        return true;
    }

    /**
     * Get the info of an user
     *
     * @access  public
     * @param   int     $id  The user ID
     * @return  mixed   Returns an array with the info of the user and false on error
     */
    function GetInfoById($id)
    {
        $params       = array();
        $params['id'] = $id;
        $sql = '
            SELECT
                [id], [username], [name], [email], [url], [createtime], [updatetime], [last_login],
                [user_type], [language], [theme], [editor], [timezone], [enabled]
            FROM [[users]]
            WHERE [id] = {id}';

        $types = array('integer', 'text', 'text', 'text', 'text', 'timestamp', 'timestamp', 'timestamp',
                       'integer', 'text', 'text', 'text', 'text', 'boolean');
        $result = $GLOBALS['db']->queryRow($sql, $params, $types);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        return $result;
    }

    /**
     * Get the info of a group
     *
     * @access  public
     * @param   int     $id  The group ID
     * @return  mixed   Returns an array with the info of the group and false on error
     */
    function GetGroupInfoById($id)
    {
        $params       = array();
        $params['id'] = $id;
        $sql = '
            SELECT
                [id], [name], [title], [description]
            FROM [[groups]]
            WHERE [id] = {id}';

        $result = $GLOBALS['db']->queryRow($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        return $result;
    }

    /**
     * Get the info of an user by the username
     *
     * @access  public
     * @param   int     $user  The username
     * @return  mixed   Returns an array with the info of the user and false on error
     */
    function GetInfoByUsername($user)
    {
        $params         = array();
        $params['user'] = strtolower($user);
        $GLOBALS['db']->dbc->loadModule('Function', null, true);
        $username = $GLOBALS['db']->dbc->function->lower('[username]');

        $sql = "
            SELECT
                [id], [username], [name], [email], [url], [user_type], [language],
                [theme], [editor], [timezone], [enabled]
            FROM [[users]]
            WHERE $username = {user}";

        return $GLOBALS['db']->queryRow($sql, $params);
    }
    
    /**
     * Returns true if a group (by its ID) can be removed
     *
     * @access  public
     * @param   int     $id  Group's ID
     * @return  boolean Can be removed?
     */
    function CanRemoveGroup($id)
    {
       $params       = array();
       $params['id'] = $id;
       
        $sql = '
            SELECT
                [removable]
            FROM [[groups]]
            WHERE [id] = {id}';

        $result = $GLOBALS['db']->queryOne($sql, $params, array('boolean'));
        if (Jaws_Error::IsError($result)) {
            return false;
        }
        
        return $result;
    }

    /**
     * Get the avatar url
     * @access public
     * @param  string   $username   Username
     * @param  string   $email      Email
     * @return string   Url to avatar image
     */
    function GetAvatar($username, $email)
    {
        $avatar = $GLOBALS['app']->getDataURL() . 'avatar/'.$username.'.png';
        if (!file_exists($avatar)) {
            require_once JAWS_PATH . 'include/Jaws/Gravatar.php';
            $avatar = Jaws_Gravatar::GetGravatar($email);
        }
        return $avatar;
    }

    /**
     * Get the info of a group by its name
     *
     * @access  public
     * @param   int     $id  The group name
     * @return  mixed   Returns an array with the info of the group and false on error
     */
    function GetGroupInfoByName($name)
    {
        $params         = array();
        $params['name'] = $name;
        $sql = '
            SELECT
                [id], [name], [title], [description]
            FROM [[groups]]
            WHERE [name] = {name}';

        $result = $GLOBALS['db']->queryRow($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        return $result;
    }

    /**
     * Get a list of all users
     *
     * @access  public
     * @param   string  $orderBy    field to order by
     * @param   string  $match      Uses a match to find users (using the username)
     * @param   string  $type       Type of user to use (0 = superadmin, 1 = admin, 2 = normal)
     * @param   boolean $onlyAdmins Only show admins
     * @return  array   Returns an array of the available users and false on error
     */
    function GetAllUsers($orderBy = 'username', $match = null, $type = 0, $onlyAdmins = false)
    {
        $fields  = array('id', 'username', 'email', 'name');
        $orderBy = strtolower($orderBy);
        if (!in_array($orderBy, $fields)) {
            if (isset($GLOBALS['log'])) {
                $GLOBALS['log']->Log(JAWS_LOG_WARNING, _t('GLOBAL_ERROR_UNKNOWN_COLUMN'));
            }
            $orderBy = 'username';
        }

        $sql = '
            SELECT
                [id], [username], [email], [url], [name], [user_type], [language],
                [theme], [editor], [timezone], [enabled]
            FROM [[users]]
            WHERE [user_type] >= {type}';
        if ($onlyAdmins == true) {
            $sql .= ' AND [user_type] < 2';
        }

        $params = array();
        if (is_string($match)) {
            $xss   = $GLOBALS['app']->loadClass('XSS', 'Jaws_XSS');
            $match = trim($match);
            $match = $xss->parse($match);

            $params['match'] = '%'.$match.'%';
            $sql.= ' AND [username] LIKE {match}';
        }
        $params['type'] = $type;
        $sql .= ' ORDER BY [' . $orderBy . ']';

        $result = $GLOBALS['db']->queryAll($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        return $result;
    }

    /**
     * Get a list of all groups
     *
     * @access  public
     * @param   string $orderBy field to order by
     * @return  array   Returns an array of the available groups and false on error
     */
    function GetAllGroups($orderBy = 'name')
    {
        $fields  = array('id', 'name', 'title');
        $orderBy = strtolower($orderBy);
        if (!in_array($orderBy, $fields)) {
            $GLOBALS['log']->Log(JAWS_LOG_WARNING, _t('GLOBAL_ERROR_UNKNOWN_COLUMN'));
            $orderBy = 'name';
        }

        $sql = '
            SELECT
                [id], [name], [title], [description]
            FROM [[groups]]
            ORDER BY [' . $orderBy. ']';

        $result = $GLOBALS['db']->queryAll($sql);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        return $result;
    }

    /**
     * Get a list of groups where a user is
     *
     * @access  public
     * @param   string  $username  Username
     * @return  array   Returns an array of the available groups and false on error
     */
    function GetGroupsOfUsername($username)
    {
        $params             = array();
        $params['username'] = $username;
        $sql = '
            SELECT
                [[groups]].[id] AS group_id, [[groups]].[name] AS group_name,
                [[users]].[id] AS user_id
            FROM [[users_groups]]
            INNER JOIN [[users]]  ON [[users]].[id] =  [[users_groups]].[user_id]
            INNER JOIN [[groups]] ON [[groups]].[id] = [[users_groups]].[group_id]
            WHERE [username] = {username}';
        $result = $GLOBALS['db']->queryAll($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        return $result;
    }

    /**
     * Get a list of groups where a user is by userid
     *
     * @access  public
     * @param   string  $user_id User ID
     * @return  array   Returns an array of the available groups and false on error
     */
    function GetGroupsOfUser($user_id)
    {
        $params            = array();
        $params['user_id'] = $user_id;
        $sql = '
            SELECT
                [[groups]].[id] AS group_id, [[groups]].[name] AS group_name
            FROM [[users_groups]]
            INNER JOIN [[groups]] ON [[groups]].[id] = [[users_groups]].[group_id]
            WHERE [[users_groups]].[user_id] = {user_id}';

        $result = $GLOBALS['db']->queryAll($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        return $result;
    }

    /**
     * Get a list of users that are inside a group
     *
     * @access  public
     * @param   string  $id  Group's ID
     * @return  array   Returns an array of the available users and false on error
     */
    function GetUsersOfGroup($id)
    {
        $params       = array();
        $params['id'] = $id;
        $sql = '
            SELECT
                [[groups]].[id] AS group_id, [[users]].[id] AS user_id, [[users]].[name] AS user_name,
                [[users]].[email] AS user_email
            FROM [[users_groups]]
            INNER JOIN [[users]]  ON [[users]].[id] =  [[users_groups]].[user_id]
            INNER JOIN [[groups]] ON [[groups]].[id] = [[users_groups]].[group_id]
            WHERE [[groups]].[id] = {id}';

        $result = $GLOBALS['db']->queryAll($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        return $result;
    }

    /**
     * Adds a new user
     *
     * @access  public
     * @param   string  $username   The username
     * @param   string  $name       User's name
     * @param   string  $email      User's email
     * @param   string  $url        User's url
     * @param   string  $passwd     User's password(md5ed)
     * @param   string  $type       User's type (ADMIN or NORMAL)
     * @return  boolean Returns true if user was sucessfully added, false if not
     */
    function AddUser($username, $name, $email, $url, $passwd, $type = 2, $enabled = true)
    {
        //We already have a $username in the DB?
        $info = $this->GetInfoByUsername($username);
        //username exists
        if (Jaws_Error::IsError($info) || isset($info['username'])) {
            return false;
        }

        if (is_numeric($type)) {
            $type = ($type > 2 && $type < 0) ? 2 : (int)$type;
        }

        MDB2::loadFile('Date');
        $params             = array();
        $params['username'] = $username;
        $params['name']     = $name;
        $params['email']    = $email;
        $params['url']      = $url;
        $params['passwd']   = $passwd;
        $params['type']     = $type;
        $params['now']      = MDB2_Date::mdbNow();
        $params['enabled']  = (bool)$enabled;

        $sql = '
            INSERT INTO [[users]]
                ([username], [name], [email], [url], [passwd], [user_type], [createtime], [updatetime], [enabled])
            VALUES
                ({username}, {name}, {email}, {url}, {passwd}, {type}, {now}, {now}, {enabled})';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return $result;
        }

        // Fetch the id of the user that was just created
        $id = $GLOBALS['db']->lastInsertID('users', 'id');
        if (Jaws_Error::IsError($id)) {
            return false;
        }

        // Let everyone know a user has been added
        $GLOBALS['app']->loadClass('Shouter', 'Jaws_EventShouter');
        $res = $GLOBALS['app']->Shouter->Shout('onAddUser', $id);
        if (Jaws_Error::IsError($res) || !$res) {
            //do nothing
        }

        return $id;
    }

    /**
     * Adds a new group
     *
     * @access  public
     * @param   string  $name        Group's name
     * @param   string  $title       Group's title
     * @param   string  $description Group's description
     * @param   boolean $removable   (Optional) Can the group be removed by users (via UI)?
     * @return  boolean Returns true if group  was sucessfully added, false if not
     */
    function AddGroup($name, $title, $description, $removable = true)
    {
        //We already have a $username in the DB?
        $info = $this->GetGroupInfoByName($name);
        if (isset($info['name'])) {
            //username exists
            return false;
        }

        $params = array();
        $params['name']        = $name;
        $params['title']       = $title;
        $params['description'] = $description;
        $params['removable']   = (is_bool($removable)) ? $removable : true;
        
        $sql = '
            INSERT INTO [[groups]]
                ([name], [title], [description], [removable])
            VALUES
                ({name}, {title}, {description}, {removable})';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        // Fetch the id of the group that was just created
        $id = $GLOBALS['db']->lastInsertID('groups', 'id');
        if (Jaws_Error::IsError($id)) {
            return false;
        }

        // Let everyone know a group has been added
        $GLOBALS['app']->loadClass('Shouter', 'Jaws_EventShouter');
        $res = $GLOBALS['app']->Shouter->Shout('onAddGroup', $id);
        if (Jaws_Error::IsError($res) || !$res) {
            //do nothing
        }

        return $id;
    }

    /**
     * Update the info of an user
     *
     * @access  public
     * @param   int     $id         User's ID
     * @param   string  $username   The username
     * @param   string  $name       User's name
     * @param   string  $email      User's email
     * @param   string  $url        User's url
     * @param   string  $passwd     User's password(md5ed)
     * @param   string  $type       Type of user to use
     * @param   boolean $enabled    enable/disable user
     * @return  boolean Returns true if user was sucessfully updated, false if not
     */
    function UpdateUser($id, $username, $name, $email, $url, $passwd, $type = null, $enabled = null)
    {
        $emptyMD5 = 'd41d8cd98f00b204e9800998ecf8427e';
        MDB2::loadFile('Date');

        $params               = array();
        $params['id']         = $id;
        $params['username']   = $username;
        $params['name']       = $name;
        $params['email']      = $email;
        $params['url']        = $url;
        $params['passwd']     = $passwd;
        $params['type']       = $type;
        $params['updatetime'] = MDB2_Date::mdbNow();
        $params['enabled']    = (bool)$enabled;

        $sql = '
            UPDATE [[users]] SET
                [username] = {username},
                [name]  = {name},
                [email] = {email},
                [url]   = {url},
                [updatetime] = {updatetime} ';
        if ($passwd != $emptyMD5) {
            $sql .= ', [passwd] = {passwd} ';
        }
        if (!is_null($type)) {
            $sql .= ', [user_type] = {type} ';
        }
        if (!is_null($enabled)) {
            $sql .= ', [enabled] = {enabled} ';
        }

        $sql .= 'WHERE [id] = {id}';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        if (isset($GLOBALS['app']->Session) && $GLOBALS['app']->Session->GetAttribute('user_id') == $id) {
            $GLOBALS['app']->Session->SetAttribute('name',     $name);
            $GLOBALS['app']->Session->SetAttribute('email',    $email);
            $GLOBALS['app']->Session->SetAttribute('url',      $url);
            $GLOBALS['app']->Session->SetAttribute('username', $username);
        }

        // Let everyone know a user has been updated
        $GLOBALS['app']->loadClass('Shouter', 'Jaws_EventShouter');
        $res = $GLOBALS['app']->Shouter->Shout('onUpdateUser', $id);
        if (Jaws_Error::IsError($res) || !$res) {
            return false;
        }

        return true;
    }

    /**
     * Update advanced options of a user such as language, theme, editor, etc..
     *
     * @access  public
     * @param   int     $id    User's ID
     * @param   array   $opts  Advanced options
     * @return  boolean Returns true on success, false on failure
     */
    function UpdateAdvancedOptions($id, $opts = array())
    {
        $validOptions = array('language', 'theme', 'editor', 'timezone');
        $params       = array();
        $updateStr    = '';
        foreach($opts as $k => $v) {
            if (in_array($k, $validOptions)) {
                $params[$k] = $v;
                $updateStr.= '['. $k . '] = {'.$k.'}, ';
            }
        }

        if (count($params) > 0) {
            $updateStr = substr($updateStr, 0, -2);
            $params['id'] = $id;
            $sql = 'UPDATE [[users]] SET '.$updateStr.
                ' WHERE [id] = {id}';
            $result = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($result)) {
                return false;
            }

            if (isset($GLOBALS['app']->Session) && $GLOBALS['app']->Session->GetAttribute('user_id') == $id) {
                foreach($params as $k => $v) {
                    $GLOBALS['app']->Session->SetAttribute($k, $v);
                }
            }
        }
        return true;
    }

    /**
     * Update the info of a group
     *
     * @access  public
     * @param   int     $id          Group's ID
     * @param   string  $name        Group's title
     * @param   string  $title       Group's name
     * @param   string  $description Group's description
     * @return  boolean Returns true if group was sucessfully updated, false if not
     */
    function UpdateGroup($id, $name, $title, $description)
    {
        $params = array();
        $params['id']          = $id;
        $params['name']        = $name;
        $params['title']       = $title;
        $params['description'] = $description;

        $sql = '
            UPDATE [[groups]] SET
                [name]        = {name},
                [title]       = {title},
                [description] = {description}
            WHERE [id] = {id}';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        // Let everyone know a group has been updated
        $GLOBALS['app']->loadClass('Shouter', 'Jaws_EventShouter');
        $res = $GLOBALS['app']->Shouter->Shout('onUpdateGroup', $id);
        if (Jaws_Error::IsError($res) || !$res) {
            //do nothing
        }

        return true;
    }

    /**
     * Deletes an user
     *
     * @access  public
     * @param   int     $id     User's ID
     * @return  boolean Returns true if user was sucessfully deleted, false if not
     */
    function DeleteUser($id)
    {
        $sql = 'SELECT COUNT([id]) FROM [[users]]';
        $c = $GLOBALS['db']->queryOne($sql);
        if (Jaws_Error::IsError($c)) {
            return false;
        }

        if ($c > '1') {
            $user = $this->GetInfoById($id);
            if (!$user) {
                return false;
            }

            $params = array();
            $params['id'] = $id;
            $sql = 'DELETE FROM [[users]] WHERE [id] = {id}';
            $result = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($result)) {
                return false;
            }

            $sql = 'DELETE FROM [[users_groups]] WHERE [user_id] = {id}';
            $result = $GLOBALS['db']->query($sql, $params);
            if (Jaws_Error::IsError($result)) {
                return false;
            }

            $GLOBALS['app']->loadClass('ACL', 'Jaws_ACL');
            $GLOBALS['app']->ACL->DeleteUserACL($user['username']);

            if (isset($GLOBALS['app']->Session)) {
                $res = $GLOBALS['app']->Session->_cache->DeleteUserSessions($id);
                if (!$res) {
                    return false;
                }
            }

            // Let everyone know that a user has been deleted
            $GLOBALS['app']->loadClass('Shouter', 'Jaws_EventShouter');
            $res = $GLOBALS['app']->Shouter->Shout('onDeleteUser', $id);
            if (Jaws_Error::IsError($res) || !$res) {
                return false;
            }

            return true;
        }

        return false;
    }


    /**
     * Deletes a group
     *
     * @access  public
     * @param   int     $id     Group's ID
     * @return  boolean Returns true if group was sucessfully deleted, false if not
     */
    function DeleteGroup($id)
    {
        if ($this->canRemoveGroup($id) === false) {
            return false;
        }
        
        $params = array();
        $params['id'] = $id;
        $sql = 'DELETE FROM [[groups]] WHERE [id] = {id}';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        $sql = 'DELETE FROM [[users_groups]] WHERE [group_id] = {id}';
        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        $GLOBALS['app']->ACL->DeleteGroupACL($id);

        // Let everyone know a group has been deleted
        $GLOBALS['app']->loadClass('Shouter', 'Jaws_EventShouter');
        $res = $GLOBALS['app']->Shouter->Shout('onDeleteGroup', $id);
        if (Jaws_Error::IsError($res) || !$res) {
            return false;
        }

        return true;
    }

    /**
     * Adds an user to a group
     *
     * @access  public
     * @param   int     $user   User's ID
     * @param   int     $group  Group's ID
     * @return  boolean Returns true if user was sucessfully added to the group, false if not
     */
    function AddUserToGroup($user, $group)
    {
        $params = array();
        $params['user']  = $user;
        $params['group'] = $group;

        $sql = '
            INSERT INTO [[users_groups]]
                ([user_id], [group_id])
            VALUES
                ({user}, {group})';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        return true;
    }

    /**
     * Deletes an user from a group
     *
     * @access  public
     * @param   int     $user   User's ID
     * @param   int     $group  Group's ID
     * @return  boolean Returns true if user was sucessfully deleted from a group, false if not
     */
    function DeleteUserFromGroup($user, $group)
    {
        $params = array();
        $params['user']  = $user;
        $params['group'] = $group;

        $sql = '
            DELETE FROM [[users_groups]]
            WHERE
                [user_id] = {user}
              AND
                [group_id] = {group}';

        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::IsError($result)) {
            return false;
        }

        return true;
    }

    /**
     * Checks if a user is in a group
     *
     * @access  public
     * @param   int     $user   User's ID
     * @param   int     $group  Group's ID
     * @return  boolean Returns true if user in in the group or false if not
     */
    function UserIsInGroup($user, $group)
    {
        $params = array();
        $params['user']  = $user;
        $params['group'] = $group;

        $sql = '
            SELECT COUNT([user_id])
            FROM [[users_groups]]
            WHERE
                [user_id] = {user}
              AND
                [group_id] = {group}';

        $howmany = $GLOBALS['db']->queryOne($sql, $params);
        if (Jaws_Error::IsError($howmany)) {
            return false;
        }

        return ($howmany == '0') ? false : true;
    }

    /**
     * Returns the ID of a user by a certain recovery key
     *
     * @access  public
     * @param   string  $key  Secret key
     * @return  boolean Success/Failure
     */
    function GetIDByRecoveryKey($key)
    {
        $key = trim($key);
        if (empty($key)) {
            return false;
        }

        $params        = array();
        $params['key'] = $key;

        $sql = ' SELECT [id] FROM [[users]] WHERE [recovery_key] = {key}';
        $uid = $GLOBALS['db']->queryOne($sql, $params);
        if (Jaws_Error::IsError($uid) || empty($uid)) {
            return false;
        }

        return $uid;
    }

    /**
     * Changes the recovery key of a certain user with a given (or auto-generated)
     * secret key (MD5)
     *
     * @access  public
     * @param   int     $uid  User's ID
     * @param   string  $key  (Optional) Secret key
     * @return  boolean Success/Failure
     */
    function ChangeRecoveryKey($uid, $key = '')
    {
        if (empty($key)) {
            $key = md5(uniqid(rand(), true)) . time() . floor(microtime()*1000);
        }

        $params        = array();
        $params['key'] = $key;
        $params['id']  = (int)$uid;

        $sql = '
            UPDATE [[users]] SET
                [recovery_key] = {key}
            WHERE [id] = {id}';
        $result = $GLOBALS['db']->query($sql, $params);
        if (Jaws_Error::isError($result)) {
            return false;
        }

        return true;
    }
}
