<?php
/**
 * Some utils functions. Random functions
 *
 * @category   JawsType
 * @package    Core
 * @author     Pablo Fischer <pablo@pablo.com.mx>
 * @author     Ali Fazelzadeh <afz@php.net>
 * @copyright  2005-2009 Jaws Development Group
 * @license    http://www.gnu.org/copyleft/lesser.html
 */
define('JAWS_OS_WIN', strtoupper(substr(PHP_OS, 0, 3)) === 'WIN');
class Jaws_Utils
{
    /**
     * Change the color of a row from a given color
     *
     * @param  string  $color  Original color(so we don't return the same color)
     * @return string  New color
     * @access public
     */
    function RowColor($color)
    {
        if ($color == '#fff') {
            return '#eee';
        }

        return '#fff';
    }

    /**
     * Create a random text
     *
     * @access  public
     * @param   int     $length Random text length
     * @return  string  The random string
     */
    function RandomText($length = 8)
    {
        include_once 'Text/Password.php';
        $word = Text_Password::create($length, 'unpronounceable', 'alphanumeric');
        return $word;
    }

    /**
     * Convert a number in bytes, kilobytes,...
     *
     * @access  public
     * @param   int     $num
     * @return  string  The converted number in string
     */
    function FormatSize($num)
    {
        $unims = array("B", "KB", "MB", "GB", "TB");
        $i = 0;
        while ($num >= 1024) {
            $i++;
            $num = $num/1024;
        }

        return number_format($num, 2). " ". $unims[$i];
    }

    /**
     * Get base url
     *
     * @access  public
     * @param   string  $base_script base script
     * @param   string rel_url relative url
     * @return  string  url of base script
     */
    function getBaseURL($base_script = 'index.php', $rel_url = false)
    {
        static $base_urls_info;
        if (!isset($base_urls_info) || !array_key_exists($base_script, $base_urls_info)) {
            $base_urls_info = array();
            $site_url = array();
            $site_url['scheme'] = (isset($_SERVER['HTTPS']) && strtolower($_SERVER['HTTPS']) == 'on')? 'https' : 'http';
            $host = $_SERVER['HTTP_HOST'];
            if (false === strpos($host, ':')) {
                $site_url['host'] = $host;
            } else {
                $site_url['host'] = substr($host, 0, strpos($host, ':'));
                $site_url['port'] = substr($host, strpos($host, ':') + 1);
            }

            $path = strip_tags($_SERVER['PHP_SELF']);
            if (false === stripos($path, $base_script)) {
                $path = strip_tags($_SERVER['SCRIPT_NAME']);
                if (false === stripos($path, $base_script)) {
                    $pInfo = isset($_SERVER['PATH_INFO'])? $_SERVER['PATH_INFO'] : '';
                    $pInfo = (empty($pInfo) && isset($_SERVER['ORIG_PATH_INFO']))? $_SERVER['ORIG_PATH_INFO'] : '';
                    $pInfo = (empty($pInfo) && isset($_ENV['PATH_INFO']))? $_ENV['PATH_INFO'] : '';
                    $pInfo = (empty($pInfo) && isset($_ENV['ORIG_PATH_INFO']))? $_ENV['ORIG_PATH_INFO'] : '';
                    $pInfo = strip_tags($pInfo);
                    if (!empty($pInfo)) {
                        $path = substr($path, 0, strpos($path, $pInfo)+1);
                    }
                }
            }

            $site_url['path'] = substr($path, 0, stripos($path, $base_script)-1);
            $base_urls_info[$base_script] = $site_url;
        } else {
            $site_url = $base_urls_info[$base_script];
        }

        if ($rel_url) {
            return $site_url['path'];
        } else {
            return $site_url['scheme']. '://'.
                   $site_url['host'].
                   (isset($site_url['port'])? (':'.$site_url['port']) : '').
                   $site_url['path'];
        }
    }

    /**
     * Get request url
     *
     * @access  public
     * @param   string  $base_script base script
     * @return  string  get url without base url
     */
    function getRequestURL($base_script = 'index.php')
    {
        static $uri;

        if (!isset($uri)) {
            if (isset($_SERVER['REQUEST_URI']) && !empty($_SERVER['REQUEST_URI'])) {
                $uri = $_SERVER['REQUEST_URI'];
            } elseif (isset($_SERVER['QUERY_STRING']) && !empty($_SERVER['QUERY_STRING'])) {
                $uri = $_SERVER['PHP_SELF'] . '?' .$_SERVER['QUERY_STRING'];
            } else {
                $uri = '';
            }

            $rel_base = Jaws_Utils::getBaseURL($base_script, true);
            $uri = htmlspecialchars(urldecode($uri), ENT_NOQUOTES, 'UTF-8');
            $uri = substr($uri, strlen($rel_base) + 1);
        }

        return $uri;
    }

    /**
     * cleaning path
     *
     * @access  public
     * @param   string  $path directory/file path
     * @return  string  The cleaned path
     */
    function clean_path($path)
    {
        $result = array();
        //$pathA = preg_split('/[\/\\\]/', $path);
        $pathA = explode('/', $path);
        if (!$pathA[0]) $result[] = '';
        foreach ($pathA AS $key => $dir) {
            if ($dir == '..') {
                if (end($result) == '..') {
                    $result[] = '..';
                } elseif (!array_pop($result)) {
                    $result[] = '..';
                }
            } elseif ($dir && $dir != '.') {
                $result[] = $dir;
            }
        }
        if (!end($pathA)) 
        $result[] = '';
        return implode('/', $result);
    }

    /**
     * relative between two path
     *
     * @access  public
     * @param   string  $path1 directory/file path
     * @param   string  $path2 directory/file path
     * @return  string  The relative path
     */
    function relative_path($path1, $path2='')
    {
        if (DIRECTORY_SEPARATOR!='/') {
            $path1 = str_replace('\\', '/', $path1);
            $path2 = str_replace('\\', '/', $path2);
        }

        //Remove starting, ending, and double / in paths
        $path1 = trim($path1,'/');
        $path2 = trim($path2,'/');
        while (substr_count($path1, '//')) $path1 = str_replace('//', '/', $path1);
        while (substr_count($path2, '//')) $path2 = str_replace('//', '/', $path2);
        //create arrays
        $arr1 = explode('/', $path1);
        if ($arr1 == array('')) $arr1 = array();
        $arr2 = explode('/', $path2);
        if ($arr2 == array('')) $arr2 = array();
        $size1 = count($arr1);
        $size2 = count($arr2);
        $path='';

        for($i=0; $i<min($size1,$size2); $i++) {
            if ($arr1[$i] != $arr2[$i]) {
                $path = '../'.$path.$arr2[$i].'/';
            }
        }
        if ($size1 > $size2) {
            for ($i = $size2; $i < $size1; $i++) {
                $path = '../'.$path;
            }
        } else {
            if ($size2 > $size1) {
                for ($i = $size1; $i < $size2; $i++) {
                    $path .= $arr2[$i].'/';
                }
            }
        }
        return $path;
    }

    /**
     * is directory writeable?
     *
     * @access  public
     * @param   string  $path directory path
     * @return  boolean True/False
     */
    function is_writable($path)
    {
        clearstatcache();
        if (!file_exists($path)) {
            return false;
        }

        /* Take care of the safe mode limitations if safe_mode=1 */
        if (ini_get('safe_mode')) {
            /* GID check */
            if (ini_get('safe_mode_gid')) {
                if (filegroup($path) == getmygid()) {
                    return (@fileperms($path) & 0020) ? is_writeable($path) : false;
                }
            } else {
                if (fileowner($path) == getmyuid()) {
                    return (@fileperms($path) & 0200) ? is_writeable($path) : false;
                }
            }
        } else {
            return is_writeable($path);
        }

        return false;
    }

    /**
     * Write a string to a file
     * @access  public
     * @see http://www.php.net/file_put_contents
     */
    function file_put_contents($file, $data, $flags = null, $resource_context = null)
    {
        $res = @file_put_contents($file, $data, $flags, $resource_context);
        if ($res !== false) {
            Jaws_Utils::chmod($file);
        }

        return $res;
    }

    /**
     * Change file/directory mode
     *
     * @access  public
     * @param   string  $path file/directory path
     * @param   integer $mode see php chmod() function
     * @return  boolean True/False
     */
    function chmod($path, $mode = null)
    {
        $result = false;
        if (is_null($mode)) {
            $php_as_owner = (function_exists('posix_getuid') && posix_getuid() === fileowner($path));
            $php_as_group = (function_exists('posix_getgid') && posix_getgid() === filegroup($path));
            if (is_dir($path)) {
                $mode = $php_as_owner? 0755 : ($php_as_group? 0775 : 0777);
            } else {
                $mode = $php_as_owner? 0644 : ($php_as_group? 0664 : 0666);
            }
        }

        $mask = umask(0);
        /* Take care of the safe mode limitations if safe_mode=1 */
        if (ini_get('safe_mode')) {
            /* GID check */
            if (ini_get('safe_mode_gid')) {
                if (filegroup($path) == getmygid()) {
                    $result = @chmod($path, $mode);
                }
            } else {
                if (fileowner($path) == getmyuid()) {
                    $result = @chmod($path, $mode);
                }
            }
        } else {
            $result = @chmod($path, $mode);
        }

        umask($mask);
        return $result;
    }

    /**
     * Make directory
     *
     * @access  public
     * @param   string  $path directory path
     * @param   integer $mode see php chmod() function
     * @return  boolean True/False
     */
    function mkdir($path, $mode = null)
    {
        $result = true;
        if (!file_exists($path) || !is_dir($path)) {
            $result = @mkdir($path);
        }

        if ($result) {
            Jaws_Utils::chmod($path, $mode);
        }

        return $result;
    }

    /**
     * Delete directories and files
     *
     * @access  public
     * @param   boolean $dirs_include
     * @param   boolean $self_include
     * @see http://www.php.net/rmdir & http://www.php.net/unlink
     */
    function Delete($path, $dirs_include = true, $self_include = true)
    {
        if (!file_exists($path)) {
            return true;
        }

        if (is_file($path) || is_link($path)) {
            // unlink can't delete read-only files in windows os
            if (JAWS_OS_WIN) {
                @chmod($path, 0777); 
            }

            return @unlink($path);
        }

        $files = scandir($path);
        foreach ($files as $file) {
            if($file == '.' || $file == '..') {
                continue;
            }

            if (!Jaws_Utils::Delete($path. DIRECTORY_SEPARATOR. $file, $dirs_include)) {
                return false;
            }
        }

        if($dirs_include && $self_include) {
            return @rmdir($path);
        }

        return true;
    }

    /**
     * Upload Files
     *
     * @access  public
     * @param   array   $files        $_FILES array
     * param    string  $dest         destination directory(include end directory separator)
     * param    string  $allowFormats permitted file format
     * param    string  $denyFiles    deny files name list, if uploaded file name in list, then rename it
     * @param   boolean $overwrite    overwite file if exist
     * @param   boolean $move_files   moving or only copying files. this param avail for non-uploaded files
     * @param   integer $max_size     max size of file
     * @return  boolean True/False
     */
    function UploadFiles($files, $dest, $allowFormats = '', $denyFiles = '', $overwrite = true, $move_files = true, $max_size = null)
    {
        if (empty($files) || !is_array($files)) {
            return new Jaws_Error(_t('GLOBAL_ERROR_UPLOAD'), 'UploadFiles');
        }

        $result = array();
        if (isset($files['name'])) {
            $files = array($files);
        }

        foreach($files as $key => $file) {
            $filename = isset($file['name']) ? $file['name'] : '';
            if (isset($file['error']) && !empty($file['error'])) {
                return new Jaws_Error(_t('GLOBAL_ERROR_UPLOAD'), 'UploadFiles');
            }

            if (empty($file['tmp_name'])) {
                continue;
            }

            if (!empty($allowFormats)) {
                $ext = end(explode('.', strtolower($filename)));
                $allowFormats = explode(',', $allowFormats);
                if (!in_array($ext, $allowFormats)) {
                    return new Jaws_Error(_t('GLOBAL_ERROR_UPLOAD_INVALID_FORMAT', $filename), 'UploadFiles');
                }
            }

            if (!empty($denyFiles)) {
                $denyFiles = explode(',', $denyFiles);
            } else {
                $denyFiles = array();
            }

            if (in_array(basename($filename), $denyFiles) ||
               (!$overwrite && file_exists($dest . $filename)))
            {
                $filename = time() . '_' . $filename;
            }
            $uploadfile = $dest . $filename;

            if (is_uploaded_file($file['tmp_name'])) {
                if (!move_uploaded_file($file['tmp_name'], $uploadfile)) {
                    return new Jaws_Error(_t('GLOBAL_ERROR_UPLOAD', $filename), 'UploadFiles');
                }
            } else {
                // On windows-systems we can't rename a file to an existing destination,
                // So we first delete destination file
                if (file_exists($uploadfile)) {
                    @unlink($uploadfile);
                }
                $res = $move_files? @rename($file['tmp_name'], $uploadfile) : @copy($file['tmp_name'], $uploadfile);
                if (!$res) {
                    return new Jaws_Error(_t('GLOBAL_ERROR_UPLOAD', $filename), 'UploadFiles');
                }
            }

            // Check if the file has been altered or is corrupted
            if (filesize($uploadfile) != $file['size']) {
                @unlink($uploadfile);
                return new Jaws_Error(_t('GLOBAL_ERROR_UPLOAD_CORRUPTED', $filename), 'UploadFiles');
            }

            Jaws_Utils::chmod($uploadfile);
            $result[$key] = $filename;
        }

        return $result;
    }

    /**
     * Get information of remote IP address
     *
     * @access  public
     * @return  array   include proxy and client ip addresses
     */
    function GetRemoteAddress()
    {
        static $proxy, $client;

        if (!isset($proxy) and !isset($client)) {
            if (!empty($_SERVER) && isset($_SERVER['REMOTE_ADDR'])) {
                $direct = $_SERVER['REMOTE_ADDR'];
            } else if (!empty($_ENV) && isset($_ENV['REMOTE_ADDR'])) {
                $direct = $_ENV['REMOTE_ADDR'];
            } else if (@getenv('REMOTE_ADDR')) {
                $direct = getenv('REMOTE_ADDR');
            }

            $proxy_flags = array('HTTP_CLIENT_IP',
                                 'HTTP_X_FORWARDED_FOR',
                                 'HTTP_X_FORWARDED',
                                 'HTTP_FORWARDED_FOR',
                                 'HTTP_FORWARDED',
                                 'HTTP_VIA',
                                 'HTTP_X_COMING_FROM',
                                 'HTTP_COMING_FROM',
                                );

            $client = '';
            foreach ($proxy_flags as $flag) {
                if (!empty($_SERVER) && isset($_SERVER[$flag])) {
                    $client = $_SERVER['HTTP_X_FORWARDED_FOR'];
                    break;
                } else if (!empty($_ENV) && isset($_ENV['HTTP_X_FORWARDED_FOR'])) {
                    $client = $_ENV['HTTP_X_FORWARDED_FOR'];
                    break;
                } else if (@getenv('HTTP_X_FORWARDED_FOR')) {
                    $client = getenv('HTTP_X_FORWARDED_FOR');
                    break;
                }
            }

            if (empty($client)) {
                $proxy  = '';
                $client = $direct;
            } else {
                $is_ip = preg_match('|^([0-9]{1,3}\.){3,3}[0-9]{1,3}|', $client, $regs);
                $client = $is_ip? $regs[0] : '';
                $proxy  = $direct;
            }

        }

        return array('proxy' => $proxy, 'client' => $client);
    }

    /**
     * Returns an array of languages
     *
     * @access  public
     * @return  array   A list of available languages
     */
    function GetLanguagesList()
    {
        static $codes;

        if (!isset($codes)) {
            $codes = array();
            $langdir = JAWS_PATH . 'languages/';
            if (!is_dir($langdir)) {
                return $codes;
            }

            $dir = scandir($langdir);
            foreach($dir as $file) {
                if ($file != '.' && $file != '..' && !strpos($file, '.php')) {
                    if (is_file($langdir.$file.'/FullName')) {
                        $fullname = implode('', @file($langdir.$file.'/FullName'));
                        if (!empty($fullname)) {
                            $codes[$file] = $fullname;
                        }
                    }
                }
            }
            asort($codes);
        }

        return $codes;
    }

}
